import 'dart:typed_data';
import 'package:flutter/material.dart';
import '../../../../layout/layout.dart';
import '../../../../../theme/colors.dart';
import 'package:image_picker/image_picker.dart';
import 'package:image_cropper/image_cropper.dart';
import '../../presenter/update_dp_presenter.dart';
import '../../../../../utils/alerts/snackbars.dart';
import '../../../../../utils/navigation/navigation.dart';
import 'package:connectivity_plus/connectivity_plus.dart';
import 'package:cached_network_image/cached_network_image.dart';
import '../../../../../utils/loading/small_loading_indicator.dart';
import 'package:flutter_image_compress/flutter_image_compress.dart';

class ChangeDp extends StatefulWidget {
  /// Variables
  final dynamic user;

  /// Initialize
  const ChangeDp({Key? key, required this.user}) : super(key: key);

  @override
  State<ChangeDp> createState() => _ChangeDpState();
}

class _ChangeDpState extends State<ChangeDp> implements UpdateDpContract {
  /// Variables
  bool processingDp = false;
  String imagePath = "";
  Uint8List? imageList;

  /// Initialize presenter
  UpdateDpPresenter? _updateDpPresenter;

  /// Initialize
  _ChangeDpState() {
    _updateDpPresenter = UpdateDpPresenter(this);
  }

  /// On username checked
  @override
  void onDpUpdated(item) async {
    if (mounted) {
      if (item['status'] == 'success') {
        /// Redirect to HOME
        BNavigator().rightNavigate(
            context,
            const Layout(
              activeIndex: 0,
            ));
      } else {
        /// Show message
        PSnackbars().showErrorSnackbar("Unable to update picture", context);
      }
    }
  }

  /// Widget
  @override
  Widget build(BuildContext context) {
    return Container(
        width: 120.0,
        height: 120.0,
        padding: const EdgeInsets.all(2.0),
        decoration: BoxDecoration(
            borderRadius: BorderRadius.circular(120.0),
            border: Border.all(color: secondaryColor, width: 2.0)),
        child: Stack(
          children: [
            /// Image
            widget.user["image"] != ""
                ? CircleAvatar(
                    backgroundImage:
                        CachedNetworkImageProvider("${widget.user["image"]}"),
                    minRadius: 60.0,
                    maxRadius: 60.0,
                  )
                : CircleAvatar(
                    backgroundColor: secondaryColor,
                    minRadius: 60.0,
                    maxRadius: 60.0,
                    child: Container(
                      padding: const EdgeInsets.only(top: 6.0),
                      child: Text(
                        "${widget.user['email'][0]}".toUpperCase(),
                        style: Theme.of(context)
                            .textTheme
                            .headlineMedium!
                            .copyWith(color: primaryColor),
                        maxLines: 1,
                        overflow: TextOverflow.ellipsis,
                      ),
                    ),
                  ),

            /// Edit button
            GestureDetector(
              onTap: () {
                if (!processingDp) {
                  _pickImage();
                }
              },
              child: CircleAvatar(
                backgroundColor: textColorDark.withOpacity(0.5),
                minRadius: 60.0,
                maxRadius: 60.0,
                child: processingDp
                    ? const SizedBox(
                        width: 24.0,
                        height: 24.0,
                        child: SmallLoadingIndicator(color: primaryColor),
                      )
                    : const Icon(
                        Icons.edit,
                        color: primaryColor,
                      ),
              ),
            )
          ],
        ));
  }

  /// Pick image
  _pickImage() async {
    final ImagePicker picker = ImagePicker();
    final XFile? image = await picker.pickImage(source: ImageSource.gallery);
    if (image?.path != null) {
      imagePath = image!.path;

      CroppedFile? croppedImage = await ImageCropper().cropImage(
          sourcePath: imagePath,
          maxWidth: 300,
          maxHeight: 300,
          aspectRatio: const CropAspectRatio(ratioX: 1, ratioY: 1));

      imageList = await FlutterImageCompress.compressWithFile(
          croppedImage!.path,
          minWidth: 300,
          minHeight: 300,
          quality: 94,
          format: CompressFormat.jpeg);
    }
    _submit();
  }

  /// Submit button
  _submit() async {
    /// Check connectivity
    final bool isConnected = await _checkConnectivity();
    if (isConnected) {
      /// If connectivity is good
      if (!processingDp && imageList != null) {
        _resetProcessing(true);
        _updateDpPresenter!.updateDp(imageList!);
      }
    } else {
      _resetProcessing(false);
      if (mounted) {
        PSnackbars().showErrorSnackbar("Not connected to Internet", context);
      }
    }
  }

  /// Reset loading
  _resetProcessing(val) {
    processingDp = val;
    _changeState();
  }

  /// Check connectivity
  _checkConnectivity() async {
    final List<ConnectivityResult> connectivityResult =
        await (Connectivity().checkConnectivity());
    if (connectivityResult.contains(ConnectivityResult.none)) {
      return false;
    } else {
      return true;
    }
  }

  /// Change state
  _changeState() {
    if (mounted) {
      setState(() {});
    }
  }
}
