import 'helpers/change_dp.dart';
import '../../../layout/layout.dart';
import '../../../../theme/colors.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import '../../../../utils/alerts/snackbars.dart';
import '../presenter/edit_profile_presenter.dart';
import '../../../../datastore/sqlite/sq_user.dart';
import '../../../../utils/inputs/input_labels.dart';
import '../../../../utils/buttons/simple_button.dart';
import '../../../../utils/navigation/navigation.dart';
import 'package:connectivity_plus/connectivity_plus.dart';
import '../../../../utils/inputs/profile_inputs/name_input.dart';

class EditProfile extends StatefulWidget {
  const EditProfile({Key? key}) : super(key: key);

  @override
  State<EditProfile> createState() => _EditProfileState();
}

class _EditProfileState extends State<EditProfile>
    implements EditProfileContract {
  /// Variables
  dynamic _user = {};
  String _name = "";
  bool _loaded = false;
  bool _hasChanged = false;
  bool _autoValidate = false;
  bool _isProcessing = false;
  final _formKey = GlobalKey<FormState>();
  final TextEditingController _nameController = TextEditingController();

  /// Initialize presenter
  EditProfilePresenter? _presenter;

  /// Initialize
  _EditProfileState() {
    _presenter = EditProfilePresenter(this);
  }

  /// On user updated
  @override
  void onUserUpdated(item) async {
    if (mounted) {
      if (item['status'] == 'success') {
        /// Redirect to HOME
        BNavigator().rightNavigate(
            context,
            const Layout(
              activeIndex: 0,
            ));
      } else {
        /// Show message
        PSnackbars().showErrorSnackbar("Error updating", context);
      }
    }
  }

  /// Init
  @override
  void initState() {
    _getData();
    super.initState();
  }

  /// Dispose
  @override
  void dispose() {
    _nameController.dispose();
    super.dispose();
  }

  /// Widget
  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        elevation: 0.0,
        leading: const BackButton(),
        title: const Text("Edit Profile"),
        systemOverlayStyle: const SystemUiOverlayStyle(
            statusBarBrightness: Brightness.light,
            statusBarIconBrightness: Brightness.dark,
            statusBarColor: primaryColor),
      ),
      body: SafeArea(
        child: Padding(
          padding: const EdgeInsets.all(16.0),
          child: Form(
            key: _formKey,
            child: Column(
              children: [
                /// Change dp
                Container(
                  height: 120.0,
                  width: double.infinity,
                  alignment: Alignment.center,
                  child: !_loaded ? Container() : ChangeDp(user: _user),
                ),

                /// Spacing
                const SizedBox(
                  height: 30.0,
                ),

                /// First name
                const InputLabel(
                  text: "Name",
                ),
                NameInput(
                  hint: "Enter your full name",
                  autoValidate: _autoValidate,
                  controller: _nameController,
                  changeFunc: () {
                    _checkChange();
                  },
                ),

                /// Spacing
                Expanded(child: Container()),

                /// Button
                SimpleButton(
                  text: "Update",
                  isProcessing: _isProcessing,
                  actionFunc: _submit,
                  color: secondaryColor,
                  textColor: primaryColor,
                ),

                /// Spacing
                const SizedBox(
                  height: 20.0,
                )
              ],
            ),
          ),
        ),
      ),
    );
  }

  /// Get data
  _getData() async {
    final List<dynamic> userInfo = await SqUser().fetchUserInfo();
    if (userInfo.isNotEmpty) {
      _user = userInfo[0];
      _name = _user['name'] ?? "";
      _nameController.text = _name;
      _loaded = true;
      _changeState();
    }
  }

  /// Submit button
  _submit() async {
    /// Check connectivity
    final bool isConnected = await _checkConnectivity();
    if (isConnected) {
      /// If connectivity is good
      if (!_isProcessing && _hasChanged) {
        _resetProcessing(true);
        _editProfile();
      }
    } else {
      if (mounted) {
        PSnackbars().showErrorSnackbar("Not connected to Internet", context);
      }
    }
  }

  /// Create profile
  _editProfile() {
    if (_formKey.currentState!.validate()) {
      /// Change to processing
      _resetProcessing(true);

      /// Get data
      final String name = _nameController.text.trim();
      _presenter!.editProfile(name);
    } else {
      _autoValidate = true;
      _resetProcessing(false);
    }
  }

  /// Check connectivity
  _checkConnectivity() async {
    final List<ConnectivityResult> connectivityResult =
        await (Connectivity().checkConnectivity());
    if (connectivityResult.contains(ConnectivityResult.none)) {
      return false;
    } else {
      return true;
    }
  }

  /// Reset loading
  _resetProcessing(val) {
    _isProcessing = val;
    _changeState();
  }

  /// Check change
  _checkChange() {
    final String currentName = _nameController.text.trim();
    if (currentName != _name) {
      _hasChanged = true;
    } else {
      _hasChanged = false;
    }
    _changeState();
  }

  /// Refresh State
  _changeState() {
    if (mounted) {
      setState(() {});
    }
  }
}
