import 'dart:io';
import 'package:huntfitnessapp/services/layout/layout.dart';
import 'package:huntfitnessapp/utils/navigation/navigation.dart';

import 'helpers/text_divider.dart';
import '../../../../theme/colors.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import '../home/helpers/login_text.dart';
import '../login/helpers/apple_button.dart';
import '../login/helpers/google_button.dart';
import '../../../../utils/search/search.dart';
import '../../../../utils/loading/loading.dart';
import '../../../../utils/alerts/snackbars.dart';
import 'package:firebase_auth/firebase_auth.dart';
import '../../../../utils/titles/simple_title.dart';
import 'package:google_sign_in/google_sign_in.dart';
import '../../../../utils/inputs/input_labels.dart';
import '../../presenter/social_login_presenter.dart';
import '../../../../utils/buttons/simple_button.dart';
import '../../presenter/email_register_presenter.dart';
import '../../../../utils/titles/simple_description.dart';
import 'package:connectivity_plus/connectivity_plus.dart';
import 'package:the_apple_sign_in/the_apple_sign_in.dart';
import '../../../../utils/inputs/auth_inputs/email_input.dart';
import '../../../../utils/inputs/auth_inputs/password_input.dart';
import '../../../../utils/inputs/auth_inputs/confirm_password.dart';

class SignUp extends StatefulWidget {
  const SignUp({super.key});

  @override
  State<SignUp> createState() => _SignUpState();
}

class _SignUpState extends State<SignUp>
    implements EmailRegisterContract, SocialLoginContract {
  /// Variables
  bool _obscure = true;
  bool _isProcessing = false;
  bool _autoValidate = false;
  final _formKey = GlobalKey<FormState>();
  final FirebaseAuth _auth = FirebaseAuth.instance;
  final GoogleSignIn _googleSignIn = GoogleSignIn();
  final TextEditingController _emailController = TextEditingController();
  final TextEditingController _passwordController = TextEditingController();
  final TextEditingController _confirmPasswordController =
      TextEditingController();

  /// Initialize presenter
  EmailRegisterPresenter? _emailPresenter;
  SocialLoginPresenter? _socialPresenter;

  /// Initialize
  _SignUpState() {
    _emailPresenter = EmailRegisterPresenter(this);
    _socialPresenter = SocialLoginPresenter(this);
  }

  /// On social login complete
  @override
  void onSocialLoginComplete(item) async {
    if (item["status"] == "success") {
      _redirect(item);
    } else {
      _resetProcessing(false);
      PSnackbars().showErrorSnackbar("Unable to log you in..", context);
    }
  }

  /// On user login complete
  @override
  void onRegisterComplete(item) async {
    if (item["status"] == "success") {
      _redirect(item);
    } else {
      _resetProcessing(false);
      PSnackbars().showErrorSnackbar("Unable to log you in..", context);
    }
  }

  /// Dispose
  @override
  void dispose() {
    _emailController.dispose();
    _passwordController.dispose();
    _confirmPasswordController.dispose();
    super.dispose();
  }

  /// Widget
  @override
  Widget build(BuildContext context) {
    return AnnotatedRegion<SystemUiOverlayStyle>(
      value: SystemUiOverlayStyle(
          statusBarColor: Colors.transparent,
          statusBarBrightness: Theme.of(context).brightness ==
              Brightness.dark
              ?Brightness.dark:Brightness.light,
          statusBarIconBrightness: Theme.of(context).brightness ==
              Brightness.dark
              ?Brightness.light:Brightness.dark),
      child: Scaffold(
        body: Stack(
          children: [
            /// Body
            SafeArea(
              child: SingleChildScrollView(
                child: Padding(
                  padding: const EdgeInsets.all(20.0),
                  child: Form(
                    key: _formKey,
                    child: Column(
                      children: [
                        /// Spacing
                        const SizedBox(
                          height: 12.0,
                        ),

                        /// Title
                        const SimpleTitle(text: "Create your Account"),

                        /// Spacing
                        const SizedBox(
                          height: 6.0,
                        ),

                        /// Description
                        const SimpleDescription(
                            text: "Your Pathway to Success"),

                        /// Spacing
                        const SizedBox(
                          height: 36.0,
                        ),

                        /// Email Input
                        const InputLabel(
                          text: "Email Address",
                        ),
                        EmailInput(
                            autoValidate: _autoValidate,
                            emailController: _emailController),

                        /// Password Input
                        const InputLabel(
                          text: "Password",
                        ),
                        PasswordInput(
                            obscure: _obscure,
                            showHide: _showHide,
                            autoValidate: _autoValidate,
                            passwordController: _passwordController),

                        /// Confirm Password Input
                        const InputLabel(
                          text: "Confirm Password",
                        ),
                        ConfirmPasswordInput(
                            autoValidate: _autoValidate,
                            confirmPasswordController:
                                _confirmPasswordController),

                        /// Register Button
                        SimpleButton(
                          text: 'Create Account',
                          actionFunc: () {
                            _register("email");
                          },
                          isProcessing: false,
                          color: secondaryColor,
                          textColor: primaryColor,
                        ),

                        /// Divider
                        const TextDivider(text: "Or Sign Up With"),

                        /// Google
                        GoogleButton(
                          loginFunc: () {
                            _register("google");
                          },
                        ),

                        /// Spacing
                        const SizedBox(
                          height: 12.0,
                        ),

                        /// Apple button
                        Platform.isIOS
                            ? AppleButton(loginFunc: () {
                                _register("apple");
                              })
                            : Container(),

                        /// Spacing
                        Platform.isIOS
                            ? const SizedBox(
                                height: 12.0,
                              )
                            : Container(),

                        /// Login text
                        const LoginText(
                          textColor: textColorDark,
                          linkColor: secondaryColor,
                        ),

                        /// Spacing
                        const SizedBox(
                          height: 26.0,
                        ),
                      ],
                    ),
                  ),
                ),
              ),
            ),

            /// Processing
            _isProcessing
                ? Positioned.fill(
                    child: Container(
                    color: textColorDark.withOpacity(0.6),
                    child: const Center(
                      child: LoadingIndicator(),
                    ),
                  ))
                : Container()
          ],
        ),
      ),
    );
  }

  /// Show or hide password
  _showHide() {
    _obscure = !_obscure;
    setState(() {});
  }

  /// Register
  _register(loginType) async {
    /// Check connectivity
    final bool isConnected = await _checkConnectivity();
    if (isConnected) {
      /// If connectivity is good
      if (!_isProcessing) {
        _resetProcessing(true);
        if (loginType == "email") {
          _registerWithEmail();
        } else if (loginType == "google") {
          _signInWithGoogle();
        } else if (loginType == "apple") {
          _signInWithApple();
        }
      }
    } else {
      if (mounted) {
        PSnackbars().showErrorSnackbar("Not connected to Internet", context);
      }
    }
  }

  /// Login
  _registerWithEmail() async {
    if (_formKey.currentState!.validate()) {
      /// Get data
      final email = _emailController.text.trim().toLowerCase();
      final password = _passwordController.text;
      final confirmPassword = _confirmPasswordController.text;

      /// Passwords don't match
      if (password != confirmPassword) {
        PSnackbars().showErrorSnackbar("Passwords do not match", context);
        _resetProcessing(false);
      } else {
        /// Register
        try {
          final res = await _auth.createUserWithEmailAndPassword(
              email: email, password: password);
          final user = res.user;
          final name = email.split("@")[0];

          /// Add data
          List<String>? search = SearchData().getLoginSearch(user!.email);
          dynamic userInfo = {
            "id": user.uid,
            "email": user.email,
            "name": name,
            "loginType": "email",
            "image": user.photoURL,
            "profileType": "personal",
            "timestamp": DateTime.now().millisecondsSinceEpoch ~/ 1000,
            "search": search
          };
          _emailPresenter!.emailRegister(userInfo);
        } catch (err) {
          _resetProcessing(false);
          _showError(err);
        }
      }
    } else {
      _autoValidate = true;
      _resetProcessing(false);
    }
  }

  /// Sign in with google
  /// function
  _signInWithApple() async {
    if (await TheAppleSignIn.isAvailable()) {
      final AuthorizationResult result = await TheAppleSignIn.performRequests([
        const AppleIdRequest(requestedScopes: [Scope.email, Scope.fullName])
      ]);
      debugPrint("${result.error}");
      switch (result.status) {
        case AuthorizationStatus.authorized:
          try {
            final AppleIdCredential? appleIdCredential = result.credential;

            OAuthProvider oAuthProvider = OAuthProvider("apple.com");
            Iterable<int> token = appleIdCredential!.identityToken ?? [];
            Iterable<int> authorizationCode =
                appleIdCredential.authorizationCode ?? [];
            final AuthCredential credential = oAuthProvider.credential(
              idToken: String.fromCharCodes(token),
              accessToken: String.fromCharCodes(authorizationCode),
            );

            final results =
                await FirebaseAuth.instance.signInWithCredential(credential);

            /// Gets user data
            final user = results.user;

            /// Social Login
            _socialLogin(user, "apple");
          } catch (e) {
            _resetProcessing(false);
          }
          break;

        case AuthorizationStatus.error:
          _resetProcessing(false);
          break;

        case AuthorizationStatus.cancelled:
          _resetProcessing(false);
          break;
      }
    } else {
      debugPrint("Apple login not available");
    }
  }

  /// Sign in with google
  /// function
  _signInWithGoogle() async {
    try {
      /// Authenticate user and get user information
      GoogleSignInAccount? googleAccount = await _googleSignIn.signIn();
      GoogleSignInAuthentication googleAuth =
          await googleAccount!.authentication;
      final AuthCredential credential = GoogleAuthProvider.credential(
          idToken: googleAuth.idToken, accessToken: googleAuth.accessToken);
      final result = await _auth.signInWithCredential(credential);
      final user = result.user;

      /// Social login
      _socialLogin(user, "google");

      /// Send data to backend
    } catch (err) {
      _resetProcessing(false);
      _showError(err);
    }
  }

  /// Social login
  _socialLogin(user, loginType) async {
    /// Get Email
    String? email = user!.email;
    if (email == null || email.isEmpty) {
      email = "";
    }

    /// Name
    String? displayName = user.displayName ?? "";
    if (displayName == "") {
      displayName = "New User";
    }

    /// Add data
    List<String> search = SearchData().getLoginSearch(email);
    dynamic userInfo = {
      "id": user.uid,
      "email": email,
      "name": displayName,
      "loginType": loginType,
      "image": user.photoURL,
      "profileType": "personal",
      "timestamp": DateTime.now().millisecondsSinceEpoch ~/ 1000,
      "search": search
    };
    _socialPresenter!.socialLogin(userInfo);
  }

  /// Show error
  _showError(err) {
    /// User already exist
    if (err is FirebaseAuthException) {
      if (err.code == 'email-already-in-use') {
        PSnackbars().showErrorSnackbar("Email is already in use.", context);
      }
    } else {
      PSnackbars().showErrorSnackbar("Error", context);
    }
  }

  /// Reset loading
  _resetProcessing(val) {
    _isProcessing = val;
    _changeState();
  }

  /// Redirect
  _redirect(item) async {
    if (mounted) {
      if (item['status'] == 'success') {
        /// Redirect
        if (item['level'] == 1) {
          /// Redirect to layout
          BNavigator().rightNavigate(context, const Layout(activeIndex: 0));
        } else {
          /// Redirect to on-boarding
          print("Redirect to Onboarding");
          BNavigator().rightNavigate(context, const Layout(activeIndex: 0));
        }
      } else {
        /// Show message
        PSnackbars().showErrorSnackbar("Error", context);
      }
    }
  }

  /// Refresh State
  _changeState() {
    if (mounted) {
      setState(() {});
    }
  }

  /// Check connectivity
  _checkConnectivity() async {
    final List<ConnectivityResult> connectivityResult =
    await (Connectivity().checkConnectivity());
    if (connectivityResult.contains(ConnectivityResult.none)) {
      return false;
    } else {
      return true;
    }
  }
}
