import 'dart:io';
import '../../../../utils/navigation/navigation.dart';
import '../../../layout/layout.dart';
import 'helpers/apple_button.dart';
import 'helpers/sign_up_text.dart';
import 'helpers/google_button.dart';
import 'helpers/forgot_pwd_link.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import '../../../../theme/colors.dart';
import '../../../../utils/search/search.dart';
import '../sign_up/helpers/text_divider.dart';
import '../../../../utils/loading/loading.dart';
import '../../../../utils/alerts/snackbars.dart';
import 'package:firebase_auth/firebase_auth.dart';
import '../../../../utils/titles/simple_title.dart';
import '../../presenter/email_login_presenter.dart';
import 'package:google_sign_in/google_sign_in.dart';
import '../../../../utils/inputs/input_labels.dart';
import '../../presenter/social_login_presenter.dart';
import '../../../../utils/buttons/simple_button.dart';
import 'package:the_apple_sign_in/the_apple_sign_in.dart';
import '../../../../utils/titles/simple_description.dart';
import 'package:connectivity_plus/connectivity_plus.dart';
import '../../../../utils/inputs/auth_inputs/email_input.dart';
import '../../../../utils/inputs/auth_inputs/password_input.dart';

class Login extends StatefulWidget {
  const Login({super.key});

  @override
  State<Login> createState() => _LoginState();
}

class _LoginState extends State<Login>
    implements EmailLoginContract, SocialLoginContract {
  /// Variables
  bool _obscure = true;
  bool _isProcessing = false;
  bool _autoValidate = false;
  final _formKey = GlobalKey<FormState>();
  final FirebaseAuth _auth = FirebaseAuth.instance;
  final GoogleSignIn _googleSignIn = GoogleSignIn();
  final TextEditingController _emailController = TextEditingController();
  final TextEditingController _passwordController = TextEditingController();

  /// Initialize presenter
  EmailLoginPresenter? _emailPresenter;
  SocialLoginPresenter? _socialPresenter;

  /// Initialize
  _LoginState() {
    _emailPresenter = EmailLoginPresenter(this);
    _socialPresenter = SocialLoginPresenter(this);
  }

  /// On social login complete
  @override
  void onSocialLoginComplete(item) async {
    if (item["status"] == "success") {
      _redirect(item);
    } else {
      _resetProcessing(false);
      PSnackbars().showErrorSnackbar("Unable to log you in..", context);
    }
  }

  /// On user login complete
  @override
  void onLoginComplete(item) async {
    if (item["status"] == "success") {
      _redirect(item);
    } else {
      _resetProcessing(false);
      PSnackbars().showErrorSnackbar("Unable to log you in..", context);
    }
  }

  /// Dispose
  @override
  void dispose() {
    _emailController.dispose();
    _passwordController.dispose();
    super.dispose();
  }

  /// Widget
  @override
  Widget build(BuildContext context) {
    return AnnotatedRegion<SystemUiOverlayStyle>(
      value: SystemUiOverlayStyle(
          statusBarColor: Colors.transparent,
          statusBarBrightness: Theme.of(context).brightness ==
              Brightness.dark
              ?Brightness.dark:Brightness.light,
          statusBarIconBrightness: Theme.of(context).brightness ==
              Brightness.dark
              ?Brightness.light:Brightness.dark),
      child: Scaffold(
        body: Stack(
          children: [
            /// Body
            SafeArea(
              child: SingleChildScrollView(
                child: Padding(
                  padding: const EdgeInsets.all(20.0),
                  child: Form(
                    key: _formKey,
                    child: Column(
                      children: [
                        /// Spacing
                        const SizedBox(
                          height: 12.0,
                        ),

                        /// Title
                        const SimpleTitle(text: "Hi, Welcome Back! 👋"),

                        /// Spacing
                        const SizedBox(
                          height: 6.0,
                        ),

                        /// Description
                        const SimpleDescription(
                            text: "Hello again, you've been missed!"),

                        /// Spacing
                        const SizedBox(
                          height: 36.0,
                        ),

                        /// Email Input
                        const InputLabel(
                          text: "Email Address",
                        ),
                        EmailInput(
                            autoValidate: _autoValidate,
                            emailController: _emailController),

                        /// Password Input
                        const InputLabel(
                          text: "Password",
                        ),
                        PasswordInput(
                            obscure: _obscure,
                            showHide: _showHide,
                            autoValidate: _autoValidate,
                            passwordController: _passwordController),

                        /// Forgot password link
                        const ForgotPasswordLink(),

                        /// Spacing
                        const SizedBox(
                          height: 32.0,
                        ),

                        /// Register Button
                        SimpleButton(
                          text: 'Login',
                          actionFunc: () {
                            _login("email");
                          },
                          isProcessing: false,
                          color: secondaryColor,
                          textColor: primaryColor,
                        ),

                        /// Divider
                        const TextDivider(text: "Or Login With"),

                        /// Google
                        GoogleButton(
                          loginFunc: () {
                            _login("google");
                          },
                        ),

                        /// Spacing
                        const SizedBox(
                          height: 12.0,
                        ),

                        /// Apple button
                        Platform.isIOS
                            ? AppleButton(loginFunc: () {
                                _login("apple");
                              })
                            : Container(),

                        /// Spacing
                        Platform.isIOS
                            ? const SizedBox(
                                height: 12.0,
                              )
                            : Container(),

                        /// Sign Up text
                        const SignUpText(
                          textColor: textColorDark,
                          linkColor: secondaryColor,
                        ),

                        /// Spacing
                        const SizedBox(
                          height: 26.0,
                        ),
                      ],
                    ),
                  ),
                ),
              ),
            ),

            /// Processing
            _isProcessing
                ? Positioned.fill(
                    child: Container(
                    color: textColorDark.withOpacity(0.6),
                    child: const Center(
                      child: LoadingIndicator(),
                    ),
                  ))
                : Container()
          ],
        ),
      ),
    );
  }

  /// Show or hide password
  _showHide() {
    _obscure = !_obscure;
    setState(() {});
  }

  /// Login
  _login(loginType) async {
    /// Check connectivity
    final bool isConnected = await _checkConnectivity();
    if (isConnected) {
      /// If connectivity is good
      if (!_isProcessing) {
        _resetProcessing(true);
        if (loginType == "email") {
          _signInWithEmail();
        } else if (loginType == "google") {
          _signInWithGoogle();
        } else if (loginType == "apple") {
          _signInWithApple();
        }
      }
    } else {
      if (mounted) {
        PSnackbars().showErrorSnackbar("Not connected to Internet", context);
      }
    }
  }

  /// Login
  _signInWithEmail() async {
    if (_formKey.currentState!.validate()) {
      /// Get data
      final email = _emailController.text.trim().toLowerCase();
      final password = _passwordController.text;

      /// Login
      try {
        final res = await _auth.signInWithEmailAndPassword(
            email: email, password: password);
        final user = res.user;

        List<String> search = SearchData().getLoginSearch(email);
        dynamic userInfo = {
          "id": user!.uid,
          "email": email,
          "loginType": "email",
          "profileType": "personal",
          "timestamp": DateTime.now().millisecondsSinceEpoch ~/ 1000,
          "search": search
        };
        _emailPresenter!.emailLogin(userInfo);
      } catch (err) {
        _resetProcessing(false);
        _showError(err);
      }
    } else {
      _autoValidate = true;
      _resetProcessing(false);
    }
  }

  /// Sign in with google
  /// function
  _signInWithApple() async {
    if (await TheAppleSignIn.isAvailable()) {
      final AuthorizationResult result = await TheAppleSignIn.performRequests([
        const AppleIdRequest(requestedScopes: [Scope.email, Scope.fullName])
      ]);

      switch (result.status) {
        case AuthorizationStatus.authorized:
          try {
            final AppleIdCredential? appleIdCredential = result.credential;

            OAuthProvider oAuthProvider = OAuthProvider("apple.com");
            Iterable<int> token = appleIdCredential!.identityToken ?? [];
            Iterable<int> authorizationCode =
                appleIdCredential.authorizationCode ?? [];
            final AuthCredential credential = oAuthProvider.credential(
              idToken: String.fromCharCodes(token),
              accessToken: String.fromCharCodes(authorizationCode),
            );

            final results =
                await FirebaseAuth.instance.signInWithCredential(credential);

            /// Gets user data
            final user = results.user;

            /// Social Login
            _socialLogin(user, "apple");
          } catch (e) {
            _resetProcessing(false);
          }
          break;

        case AuthorizationStatus.error:
          _resetProcessing(false);
          break;

        case AuthorizationStatus.cancelled:
          _resetProcessing(false);
          break;
      }
    }
  }

  /// Sign in with google
  /// function
  _signInWithGoogle() async {
    try {
      /// Authenticate user and get user information
      GoogleSignInAccount? googleAccount = await _googleSignIn.signIn();
      GoogleSignInAuthentication googleAuth =
          await googleAccount!.authentication;
      final AuthCredential credential = GoogleAuthProvider.credential(
          idToken: googleAuth.idToken, accessToken: googleAuth.accessToken);
      final result = await _auth.signInWithCredential(credential);
      final user = result.user;

      /// Social login
      _socialLogin(user, "google");

      /// Send data to backend
    } catch (err) {
      _resetProcessing(false);
      _showError(err);
    }
  }

  /// Social login
  _socialLogin(user, loginType) async {
    /// Split Email
    String? email = user!.email;
    if (email == null || email.isEmpty) {
      email = "";
    }

    /// Name
    String? displayName = user.displayName ?? "";
    if (displayName == "") {
      displayName = "New User";
    }

    /// Add data
    List<String> search = SearchData().getLoginSearch(email);
    dynamic userInfo = {
      "id": user.uid,
      "email": email,
      "name": displayName,
      "loginType": loginType,
      "image": user.photoURL,
      "profileType": "personal",
      "timestamp": DateTime.now().millisecondsSinceEpoch ~/ 1000,
      "search": search
    };
    _socialPresenter!.socialLogin(userInfo);
  }

  /// Show error
  _showError(err) {
    if (err is FirebaseAuthException) {
      if (err.code == 'user-not-found') {
        PSnackbars().showErrorSnackbar("User not found", context);
      } else if (err.code == 'INVALID_LOGIN_CREDENTIALS') {
        PSnackbars().showErrorSnackbar("Incorrect email or password", context);
      } else {
        PSnackbars().showErrorSnackbar("Error", context);
      }
    } else {
      PSnackbars().showErrorSnackbar("Error", context);
    }
  }

  /// Reset loading
  _resetProcessing(val) {
    _isProcessing = val;
    _changeState();
  }

  /// Redirect
  _redirect(item) {
    if (mounted) {
      if (item['status'] == 'success') {
        /// Redirect
        if (item['level'] == 1) {
          /// Redirect to layout
          BNavigator().rightNavigate(context, const Layout(activeIndex: 0));
        } else {
          /// Redirect to on-boarding
          print("Redirect to Onboarding");
          BNavigator().rightNavigate(context, const Layout(activeIndex: 0));
        }
      } else {
        /// Show message
        PSnackbars().showErrorSnackbar("Error", context);
      }
    }
  }

  /// Refresh State
  _changeState() {
    if (mounted) {
      setState(() {});
    }
  }

  /// Check connectivity
  _checkConnectivity() async {
    final List<ConnectivityResult> connectivityResult =
    await (Connectivity().checkConnectivity());
    if (connectivityResult.contains(ConnectivityResult.none)) {
      return false;
    } else {
      return true;
    }
  }
}
