import 'email_sent.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'helpers/send_email_button.dart';
import '../../../../theme/device_type.dart';
import '../../../../utils/alerts/snackbars.dart';
import 'package:firebase_auth/firebase_auth.dart';
import '../../../../utils/inputs/input_labels.dart';
import '../../../../utils/navigation/navigation.dart';
import 'package:connectivity_plus/connectivity_plus.dart';
import '../../../../utils/inputs/auth_inputs/email_input.dart';

class ForgotPassword extends StatefulWidget {
  const ForgotPassword({super.key});

  @override
  ForgotPasswordState createState() => ForgotPasswordState();
}

class ForgotPasswordState extends State<ForgotPassword> {
  /// Variables
  bool _mounted = true;
  bool _isProcessing = false;
  bool _autoValidate = false;
  final _formKey = GlobalKey<FormState>();
  final FirebaseAuth _auth = FirebaseAuth.instance;
  final GlobalKey<ScaffoldState> _scaffoldState = GlobalKey();
  final TextEditingController _emailController = TextEditingController();

  /// Dispose
  @override
  void dispose() {
    _mounted = false;
    _emailController.dispose();
    super.dispose();
  }

  /// Widget
  @override
  Widget build(BuildContext context) {
    return AnnotatedRegion<SystemUiOverlayStyle>(
      value: SystemUiOverlayStyle(
          statusBarColor: Colors.transparent,
          statusBarBrightness: Theme.of(context).brightness ==
              Brightness.dark
              ?Brightness.dark:Brightness.light,
          statusBarIconBrightness: Theme.of(context).brightness ==
              Brightness.dark
              ?Brightness.light:Brightness.dark),
      child: Scaffold(
        key: _scaffoldState,
        appBar: AppBar(
          elevation: 0.0,
          leading: const CloseButton(),
        ),
        body: SingleChildScrollView(
          physics: const AlwaysScrollableScrollPhysics(),
          child: Container(
            padding: const EdgeInsets.all(20.0),
            child: Form(
              key: _formKey,
              child: Column(
                children: [
                  /// Image
                  Container(
                    width: double.infinity,
                    height: Device.screenHeight / 4.0,
                    alignment: Alignment.center,
                    padding: const EdgeInsets.only(left: 20.0),
                    child: Image.asset(
                      'assets/vectors/forgot_password.png',
                      fit: BoxFit.contain,
                    ),
                  ),

                  /// Spacing
                  const SizedBox(
                    height: 20.0,
                  ),

                  /// Title
                  Text(
                    "Forgot Password?",
                    textAlign: TextAlign.center,
                    style: Theme.of(context).textTheme.titleLarge,
                  ),

                  /// Spacing
                  const SizedBox(
                    height: 5.0,
                  ),

                  /// Subtitle
                  Padding(
                    padding: Device.screenWidth > 360.0
                        ? const EdgeInsets.all(10.0)
                        : const EdgeInsets.symmetric(vertical: 10.0),
                    child: Text(
                      "Please enter your registered email to reset your password",
                      textAlign: TextAlign.center,
                      style: Theme.of(context).textTheme.bodyLarge,
                    ),
                  ),

                  /// Spacing
                  const SizedBox(
                    height: 20.0,
                  ),

                  /// Email Input
                  const InputLabel(
                    text: "Email",
                  ),
                  EmailInput(
                    autoValidate: _autoValidate,
                    emailController: _emailController,
                  ),

                  /// Spacing
                  const SizedBox(
                    height: 10.0,
                  ),

                  /// Send email
                  SendEmailButton(
                    screenWidth: Device.screenWidth,
                    actionFunc: _validate,
                    isProcessing: _isProcessing,
                  )
                ],
              ),
            ),
          ),
        ),
      ),
    );
  }

  /// Validate
  _validate() async {
    /// Check connectivity
    final bool isConnected = await _checkConnectivity();
    if (isConnected) {
      /// If connectivity is good
      if (!_isProcessing) {
        _resetProcessing(true);
        if (_formKey.currentState!.validate()) {
          _sendEmail();
        } else {
          _autoValidate = true;
          _resetProcessing(false);
        }
      }
    } else {
      if (mounted) {
        PSnackbars().showErrorSnackbar("Not connected to Internet", context);
      }
    }
  }

  /// Send Email
  _sendEmail() async {
    final email = _emailController.text.trim().toLowerCase();
    try {
      _auth.sendPasswordResetEmail(email: email);
      if (mounted) {
        BNavigator().rightNavigate(context, EmailSent(email: email));
      }
    } on FirebaseAuthException catch (e) {
      debugPrint("$e");
      PSnackbars().showErrorSnackbar("Error", context);
    }
  }

  /// Check connectivity
  _checkConnectivity() async {
    final List<ConnectivityResult> connectivityResult =
    await (Connectivity().checkConnectivity());
    if (connectivityResult.contains(ConnectivityResult.none)) {
      return false;
    } else {
      return true;
    }
  }

  /// Reset loading
  _resetProcessing(val) {
    _isProcessing = val;
    _changeState();
  }

  /// Refresh State
  _changeState() {
    if (_mounted) {
      setState(() {});
    }
  }
}
