// 若為白底黑線 (一般跑道) 請設為 true
// 若為黑底白線 (特殊跑道) 請設為 false
const boolean BLACK_LINE = true; 

// 馬達驅動板定義
// 馬達 A (左輪)
#define A_1A  9   // PWM 接腳
#define A_1B  10  // DIR 接腳

// 馬達 B (右輪)
#define B_1A  5   // PWM 接腳
#define B_1B  6   // DIR 接腳

// 循線模組定義 (雙感測器)
#define SensorLeft    11  // 左感測器
#define SensorRight   12  // 右感測器

// PWM 速度設定
const int SPEED_FORWARD = 100;  // 直行速度
const int SPEED_TURN    = 100;  // 修正速度
const int LOOP_DELAY_MS = 10;   // 每次循線的迴圈延遲，避免過度抖動
const int LEFT_DIR      = -1;   // 左輪方向校正：若接線反向，設為 -1；正常為 1

// 全域變數：根據跑道顏色決定感測器讀到線/背景的 HIGH/LOW
int VAL_LINE;  // 線的感測值 (HIGH 或 LOW)
int VAL_BG;    // 背景的感測值 (LOW 或 HIGH)

// 將單輪的速度與方向寫入驅動板
// speed > 0 前進，speed < 0 後退，speed = 0 停止
void writeWheel(int pwmPin, int dirPin, int speed) {
  int pwm = constrain(abs(speed), 0, 255);
  if (speed > 0) {
    analogWrite(pwmPin, pwm);
    digitalWrite(dirPin, LOW);
  } else if (speed < 0) {
    digitalWrite(pwmPin, LOW);
    analogWrite(dirPin, pwm);
  } else {
    digitalWrite(pwmPin, LOW);
    digitalWrite(dirPin, LOW);
  }
}

// 同時控制左右輪，讓程式碼更精簡
void drive(int leftSpeed, int rightSpeed) {
  writeWheel(A_1A, A_1B, leftSpeed * LEFT_DIR);
  writeWheel(B_1A, B_1B, rightSpeed);
}

void setup() {
  Serial.begin(9600);
  
  // 設定循線模式參數
  if (BLACK_LINE) {
    // 白底黑線模式: 線是黑色(不反光=HIGH), 背景是白色(反光=LOW)
    VAL_LINE = HIGH;
    VAL_BG = LOW;
  } else {
    // 黑底白線模式: 線是白色(反光=LOW), 背景是黑色(不反光=HIGH)
    VAL_LINE = LOW;
    VAL_BG = HIGH;
  }

  // 初始化馬達接腳
  pinMode(A_1A, OUTPUT);
  pinMode(A_1B, OUTPUT);
  pinMode(B_1A, OUTPUT);
  pinMode(B_1B, OUTPUT);
  
  // 初始化循線感測器接腳
  pinMode(SensorLeft,   INPUT); 
  pinMode(SensorRight,  INPUT);
}

// 循線邏輯 (雙感測器) 
void trackingLogic() {
  // 讀取感測器狀態：只要有感測到「線的顏色」就視為踩到線
  bool isLeft  = (digitalRead(SensorLeft)  == VAL_LINE);
  bool isRight = (digitalRead(SensorRight) == VAL_LINE);

  /* 
   * 判斷方式 (感測器在黑線左右兩側)：
   * 1. !isLeft && !isRight -> 線在中間，保持直行
   * 2.  isLeft && !isRight -> 左感測到線，車體偏右，向左修正
   * 3. !isLeft &&  isRight -> 右感測到線，車體偏左，向右修正
   * 4.  isLeft &&  isRight -> 兩邊都在線上 (橫線/十字) 直接穿過
   */
  if (!isLeft && !isRight) {
    drive(SPEED_FORWARD, SPEED_FORWARD);          // 直行
  } else if (isLeft && !isRight) {
    drive(-SPEED_TURN, SPEED_TURN);               // 左輪退/停，右輪進，向左修
  } else if (!isLeft && isRight) {
    drive(SPEED_TURN, -SPEED_TURN);               // 右輪退/停，左輪進，向右修
  } else {
    drive(SPEED_FORWARD, SPEED_FORWARD);          // 同時踩到線，直接前進通過
  }
}

// 主程式迴圈
void loop() {
  trackingLogic();
  delay(LOOP_DELAY_MS); // 避免過度頻繁寫入馬達，提升穩定度
}

