/**
 * Fliz Video Generator - Scripts Frontend
 *
 * @package Fliz_Video_Generator
 */

/* global jQuery, flizVideo */

(function($) {
    'use strict';

    var FlizVideo = {
        pollInterval: 30000, // 30 secondes
        intervals: {},

        /**
         * Initialisation
         */
        init: function() {
            this.initPolling();
        },

        /**
         * Initialiser le polling pour les vidéos en chargement
         */
        initPolling: function() {
            var self = this;

            $('.fliz-video-loading').each(function() {
                var $wrapper = $(this);
                var videoId = $wrapper.data('video-id');

                if (videoId) {
                    self.startPolling(videoId, $wrapper);
                }
            });
        },

        /**
         * Démarrer le polling pour une vidéo
         */
        startPolling: function(videoId, $wrapper) {
            var self = this;

            // Vérifier immédiatement
            this.checkStatus(videoId, $wrapper);

            // Puis périodiquement
            this.intervals[videoId] = setInterval(function() {
                self.checkStatus(videoId, $wrapper);
            }, this.pollInterval);
        },

        /**
         * Vérifier le statut d'une vidéo
         */
        checkStatus: function(videoId, $wrapper) {
            var self = this;

            $.ajax({
                url: flizVideo.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fliz_check_video',
                    security: flizVideo.nonce,
                    video_id: videoId
                },
                success: function(response) {
                    if (response.success) {
                        var status = response.data.status;

                        if (status === 'complete' && response.data.url) {
                            self.stopPolling(videoId);
                            self.showVideo($wrapper, response.data.url);
                        } else if (['failed', 'failed_unrecoverable', 'block'].indexOf(status) !== -1) {
                            self.stopPolling(videoId);
                            $wrapper.remove();
                        }
                        // Sinon, continuer le polling
                    }
                },
                error: function() {
                    // En cas d'erreur, continuer le polling silencieusement
                }
            });
        },

        /**
         * Arrêter le polling
         */
        stopPolling: function(videoId) {
            if (this.intervals[videoId]) {
                clearInterval(this.intervals[videoId]);
                delete this.intervals[videoId];
            }
        },

        /**
         * Afficher la vidéo
         */
        showVideo: function($wrapper, url) {
            var html = '<video class="fliz-video-player" controls preload="metadata" playsinline>' +
                '<source src="' + this.escapeHtml(url) + '" type="video/mp4">' +
                '</video>';

            $wrapper
                .removeClass('fliz-video-loading')
                .html(html);
        },

        /**
         * Échapper le HTML
         */
        escapeHtml: function(text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    // Initialiser quand le DOM est prêt
    $(document).ready(function() {
        FlizVideo.init();
    });

})(jQuery);
