<?php
/**
 * Classe Fliz_Shortcode
 *
 * Gère le shortcode d'affichage des vidéos.
 *
 * @package Fliz_Video_Generator
 */

// Empêcher l'accès direct
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Classe Fliz_Shortcode
 */
class Fliz_Shortcode {

    /**
     * Initialiser
     */
    public function init() {
        add_shortcode( 'fliz_video', array( $this, 'render_shortcode' ) );
        add_action( 'wp_ajax_fliz_check_video', array( $this, 'ajax_check_video' ) );
        add_action( 'wp_ajax_nopriv_fliz_check_video', array( $this, 'ajax_check_video' ) );
    }

    /**
     * Afficher le shortcode
     *
     * @param array $atts Attributs du shortcode.
     * @return string HTML.
     */
    public function render_shortcode( $atts ) {
        $atts = shortcode_atts(
            array(
                'id'     => '',
                'width'  => '100%',
                'height' => 'auto',
            ),
            $atts,
            'fliz_video'
        );

        $video_id = sanitize_text_field( $atts['id'] );
        if ( empty( $video_id ) ) {
            return '';
        }

        // Vérifier le cache local
        global $post;
        if ( $post ) {
            $cached_status = get_post_meta( $post->ID, '_fliz_video_status', true );
            $cached_url    = get_post_meta( $post->ID, '_fliz_video_url', true );

            if ( 'complete' === $cached_status && ! empty( $cached_url ) ) {
                $output = $this->render_video_player( $cached_url, $atts );
                return apply_filters( 'fliz_shortcode_output', $output, $atts, array( 'status' => 'complete', 'url' => $cached_url ) );
            }

            if ( 'error' === $cached_status ) {
                return '';
            }
        }

        // Vérifier le statut de la vidéo via l'API
        $client = new Fliz_API_Client();
        $result = $client->get_video_status( $video_id );

        if ( is_wp_error( $result ) ) {
            return '';
        }

        $status = isset( $result['status'] ) ? $result['status'] : 'unknown';
        $url    = isset( $result['url'] ) ? $result['url'] : '';

        if ( 'complete' === $status && ! empty( $url ) ) {
            // Mettre à jour le cache
            if ( $post ) {
                update_post_meta( $post->ID, '_fliz_video_url', esc_url_raw( $url ) );
                update_post_meta( $post->ID, '_fliz_video_status', 'complete' );
            }

            $output = $this->render_video_player( $url, $atts );
            return apply_filters( 'fliz_shortcode_output', $output, $atts, $result );
        } elseif ( ! in_array( $status, array( 'failed', 'failed_unrecoverable', 'block', 'complete' ), true ) ) {
            // Vidéo en cours de génération
            $output = $this->render_loading_state( $video_id, $atts );
            return apply_filters( 'fliz_shortcode_output', $output, $atts, $result );
        }

        return '';
    }

    /**
     * Afficher le player vidéo
     *
     * @param string $url  URL de la vidéo.
     * @param array  $atts Attributs.
     * @return string HTML.
     */
    private function render_video_player( $url, $atts ) {
        $width  = esc_attr( $atts['width'] );
        $height = esc_attr( $atts['height'] );

        ob_start();
        ?>
        <div class="fliz-video-wrapper" style="width: <?php echo esc_attr( $width ); ?>;">
            <video 
                class="fliz-video-player"
                controls
                preload="metadata"
                style="width: 100%; height: <?php echo esc_attr( $height ); ?>;"
                playsinline
            >
                <source src="<?php echo esc_url( $url ); ?>" type="video/mp4">
                <?php esc_html_e( 'Your browser does not support the video tag.', 'fliz-video-generator' ); ?>
            </video>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Afficher l'état de chargement
     *
     * @param string $video_id ID de la vidéo.
     * @param array  $atts     Attributs.
     * @return string HTML.
     */
    private function render_loading_state( $video_id, $atts ) {
        $width = esc_attr( $atts['width'] );

        ob_start();
        ?>
        <div 
            class="fliz-video-wrapper fliz-video-loading" 
            style="width: <?php echo esc_attr( $width ); ?>;"
            data-video-id="<?php echo esc_attr( $video_id ); ?>"
        >
            <div class="fliz-video-loading-content">
                <div class="fliz-video-spinner"></div>
                <p><?php esc_html_e( 'Video is being generated...', 'fliz-video-generator' ); ?></p>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Vérifier le statut via AJAX (public)
     */
    public function ajax_check_video() {
        check_ajax_referer( 'fliz_video_nonce', 'security' );

        $video_id = isset( $_POST['video_id'] ) ? sanitize_text_field( wp_unslash( $_POST['video_id'] ) ) : '';
        if ( empty( $video_id ) ) {
            wp_send_json_error();
        }

        $client = new Fliz_API_Client();
        $result = $client->get_video_status( $video_id );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error();
        }

        wp_send_json_success( $result );
    }
}
