<?php
/**
 * Classe Fliz_Logger
 *
 * Gère les logs de génération de vidéos.
 *
 * @package Fliz_Video_Generator
 */

// Empêcher l'accès direct
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Classe Fliz_Logger
 */
class Fliz_Logger {

	/**
	 * Nom de la table (sans préfixe)
	 *
	 * @var string
	 */
	const TABLE_NAME = 'fliz_video_logs';

	/**
	 * Nom de la table complet
	 *
	 * @return string
	 */
	private static function get_table_name() {
		global $wpdb;
		return $wpdb->prefix . self::TABLE_NAME;
	}

	/**
	 * Créer un log
	 *
	 * @param int    $post_id  ID de l'article.
	 * @param string $video_id ID de la vidéo.
	 * @param string $status   Statut.
	 * @param string $message  Message.
	 * @return int|false ID du log ou false.
	 */
	public static function log( $post_id, $video_id, $status, $message = '' ) {
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$result = $wpdb->insert(
			self::get_table_name(),
			array(
				'post_id'    => absint( $post_id ),
				'video_id'   => sanitize_text_field( $video_id ),
				'status'     => sanitize_text_field( $status ),
				'message'    => sanitize_textarea_field( $message ),
				'created_at' => current_time( 'mysql' ),
				'updated_at' => current_time( 'mysql' ),
			),
			array( '%d', '%s', '%s', '%s', '%s', '%s' )
		);

		return $result ? $wpdb->insert_id : false;
	}

	/**
	 * Mettre à jour le statut
	 *
	 * @param string $video_id ID de la vidéo.
	 * @param string $status   Nouveau statut.
	 * @param string $message  Message.
	 * @return int|false Nombre de lignes modifiées.
	 */
	public static function update_status( $video_id, $status, $message = '' ) {
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		return $wpdb->update(
			self::get_table_name(),
			array(
				'status'     => sanitize_text_field( $status ),
				'message'    => sanitize_textarea_field( $message ),
				'updated_at' => current_time( 'mysql' ),
			),
			array( 'video_id' => sanitize_text_field( $video_id ) ),
			array( '%s', '%s', '%s' ),
			array( '%s' )
		);
	}

	/**
	 * Récupérer les logs
	 *
	 * @param array $args Arguments de requête.
	 * @return array
	 */
	public static function get_logs( $args = array() ) {
		global $wpdb;

		$defaults = array(
			'post_id' => null,
			'status'  => null,
			'limit'   => 100,
			'offset'  => 0,
			'orderby' => 'created_at',
			'order'   => 'DESC',
		);

		$args = wp_parse_args( $args, $defaults );

		// Valider et échapper orderby (whitelist + esc_sql)
		$allowed_orderby = array( 'created_at', 'updated_at', 'status' );
		$orderby         = in_array( $args['orderby'], $allowed_orderby, true ) ? $args['orderby'] : 'created_at';
		$orderby         = esc_sql( $orderby );

		// Valider order (whitelist + esc_sql)
		$order = 'DESC' === strtoupper( $args['order'] ) ? 'DESC' : 'ASC';
		$order = esc_sql( $order );

		// Sanitizer les valeurs
		$limit   = absint( $args['limit'] );
		$offset  = absint( $args['offset'] );
		$post_id = ! is_null( $args['post_id'] ) ? absint( $args['post_id'] ) : null;
		$status  = ! is_null( $args['status'] ) ? sanitize_text_field( $args['status'] ) : null;

		// Construire la requête avec prepare
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter -- $orderby and $order are whitelisted and escaped with esc_sql().
		$table_name = $wpdb->prefix . self::TABLE_NAME;

		if ( ! is_null( $post_id ) && ! is_null( $status ) ) {
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM `{$table_name}` WHERE post_id = %d AND status = %s ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d",
					$post_id,
					$status,
					$limit,
					$offset
				)
			);
		} elseif ( ! is_null( $post_id ) ) {
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM `{$table_name}` WHERE post_id = %d ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d",
					$post_id,
					$limit,
					$offset
				)
			);
		} elseif ( ! is_null( $status ) ) {
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM `{$table_name}` WHERE status = %s ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d",
					$status,
					$limit,
					$offset
				)
			);
		} else {
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM `{$table_name}` ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d",
					$limit,
					$offset
				)
			);
		}
		// phpcs:enable

		return $results;
	}

	/**
	 * Compter les logs par statut
	 *
	 * @return array
	 */
	public static function count_by_status() {
		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$results = $wpdb->get_results( "SELECT status, COUNT(*) as count FROM `{$table_name}` GROUP BY status" );
		// phpcs:enable

		$counts = array(
			'all'      => 0,
			'pending'  => 0,
			'complete' => 0,
			'error'    => 0,
		);

		foreach ( $results as $row ) {
			if ( isset( $counts[ $row->status ] ) ) {
				$counts[ $row->status ] = (int) $row->count;
			}
			$counts['all'] += (int) $row->count;
		}

		return $counts;
	}

	/**
	 * Supprimer un log
	 *
	 * @param int $log_id ID du log.
	 * @return bool
	 */
	public static function delete( $log_id ) {
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		return $wpdb->delete(
			self::get_table_name(),
			array( 'id' => absint( $log_id ) ),
			array( '%d' )
		);
	}

	/**
	 * Récupérer un log par video_id
	 *
	 * @param string $video_id ID de la vidéo.
	 * @return object|null
	 */
	public static function get_by_video_id( $video_id ) {
		global $wpdb;

		$table_name = $wpdb->prefix . self::TABLE_NAME;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$result = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM `{$table_name}` WHERE video_id = %s ORDER BY created_at DESC LIMIT 1",
				sanitize_text_field( $video_id )
			)
		);
		// phpcs:enable

		return $result;
	}
}
