<?php
/**
 * Client API Fliz
 *
 * @package Fliz_Video_Generator
 */

// Empêcher l'accès direct
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Classe Fliz_API_Client
 *
 * Client pour l'API REST Fliz v1.0.0
 * Documentation : https://app.fliz.ai/api-docs
 */
class Fliz_API_Client {

	/**
     * URL de base de l'API
	 */
	private const API_BASE_URL = 'https://app.fliz.ai';

	/**
     * Clé API (JWT Token)
	 *
	 * @var string
	 */
	private $api_key;

	/**
     * Timeout par défaut (secondes)
	 *
	 * @var int
	 */
	private $timeout = 30;

	/**
     * Constructeur
	 *
	 * @param string|null $api_key Clé API optionnelle.
	 */
	public function __construct( $api_key = null ) {
		$this->api_key = $api_key ?? $this->get_stored_api_key();
	}

	/**
     * Récupérer la clé API stockée
	 *
	 * @return string
	 */
	private function get_stored_api_key() {
		$options = get_option( 'fliz_settings', array() );
		return isset( $options['api_key'] ) ? $options['api_key'] : '';
	}

	/**
     * Effectuer une requête API
	 *
	 * @param string $method   Méthode HTTP (GET, POST).
	 * @param string $endpoint Endpoint API (ex: /api/rest/video).
	 * @param array  $data     Données à envoyer (pour POST).
	 * @return array|WP_Error
	 */
	private function request( $method, $endpoint, $data = array() ) {
		if ( empty( $this->api_key ) ) {
			return new WP_Error(
				'fliz_no_api_key',
				__( 'API key is not configured.', 'fliz-video-generator' )
			);
		}

		$url = self::API_BASE_URL . $endpoint;

		$args = array(
			'method'  => $method,
			'headers' => array(
				'Authorization' => 'Bearer ' . $this->api_key,
				'Content-Type'  => 'application/json',
			),
			'timeout' => $this->timeout,
		);

		if ( ! empty( $data ) && 'POST' === $method ) {
			$args['body'] = wp_json_encode( $data );
		}

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( $code < 200 || $code >= 300 ) {
			$message = isset( $body['error'] )
				? $body['error']
				: __( 'API request failed.', 'fliz-video-generator' );

			// Détecter les erreurs d'authentification (JWT invalide)
			$is_auth_error = ( 401 === $code || 403 === $code )
				|| ( is_string( $message ) && ( strpos( $message, 'JWT' ) !== false || strpos( $message, 'verify' ) !== false || strpos( $message, 'token' ) !== false ) );

			if ( $is_auth_error ) {
				return new WP_Error(
					'fliz_invalid_api_key',
					__( 'Invalid API key. Please check your API key in the plugin settings.', 'fliz-video-generator' ),
					array(
						'status'       => $code,
						'need_config'  => true,
						'settings_url' => admin_url( 'admin.php?page=fliz-settings' ),
					)
				);
			}

			return new WP_Error( 'fliz_api_error', $message, array( 'status' => $code ) );
		}

		return $body;
	}

	/**
     * Tester la connexion API
	 *
     * Utilise l'endpoint voices pour vérifier que le token est valide
	 *
	 * @return array|WP_Error
	 */
	public function test_connection() {
		$result = $this->get_voices();

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return array(
			'success' => true,
			'message' => __( 'Connection successful!', 'fliz-video-generator' ),
		);
	}

	/**
     * Créer une vidéo
	 *
	 * POST /api/rest/video
	 *
	 * @param array $params Paramètres de création.
	 * @return array|WP_Error Contient video_id et cost en cas de succès.
	 */
	public function create_video( $params ) {
        // Valeurs par défaut
		$defaults = array(
			'name'              => '',
			'description'       => '',
			'category'          => 'article',
			'format'            => 'size_16_9',
			'lang'              => 'en',
			'script_style'      => null,
			'image_style'       => null,
			'image_urls'        => null,
			'caption_style'     => 'animated_background',
			'caption_position'  => 'bottom',
			'caption_font'      => 'poppins',
			'caption_color'     => null,
			'caption_uppercase' => null,
			'voice_id'          => null,
			'is_male_voice'     => null,
			'music_id'          => null,
			'music_url'         => null,
			'music_volume'      => null,
			'watermark_url'     => null,
			'site_url'          => null,
			'site_name'         => null,
			'webhook_url'       => null,
			'is_automatic'      => true,
		);

		// Sauvegarder watermark_url AVANT wp_parse_args car null est une valeur valide pour "no watermark"
		// Note: isset() retourne false pour null, donc on utilise array_key_exists()
		$has_explicit_watermark_null = array_key_exists( 'watermark_url', $params ) && null === $params['watermark_url'];

		$params = wp_parse_args( $params, $defaults );

        // Nettoyer les valeurs null pour ne pas les envoyer
		$clean_params = array_filter(
			$params,
			function ( $value ) {
				return ! is_null( $value ) && '' !== $value;
			}
		);

		// Restaurer watermark_url si explicitement défini à null (pour "no watermark")
		// - null = pas de watermark
		// - non défini = watermark Fliz par défaut
		// - URL = watermark personnalisé
		if ( $has_explicit_watermark_null ) {
			$clean_params['watermark_url'] = null;
		}

		$body = array(
			'fliz_video_create_input' => $clean_params,
		);

		$result = $this->request( 'POST', '/api/rest/video', $body );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

        // Extraire les données de la réponse
		if ( isset( $result['fliz_video_create'] ) ) {
			return array(
				'video_id' => $result['fliz_video_create']['video_id'],
				'cost'     => $result['fliz_video_create']['cost'],
			);
		}

        return new WP_Error( 'fliz_invalid_response', __( 'Invalid API response.', 'fliz-video-generator' ) );
	}

	/**
     * Récupérer une vidéo par ID
	 *
	 * GET /api/rest/videos/{id}
	 *
	 * @param string $video_id UUID de la vidéo.
	 * @return array|WP_Error
	 */
	public function get_video( $video_id ) {
		$video_id = sanitize_text_field( $video_id );

        // Valider le format UUID
		if ( ! preg_match( '/^[a-f0-9]{8}-[a-f0-9]{4}-4[a-f0-9]{3}-[89ab][a-f0-9]{3}-[a-f0-9]{12}$/i', $video_id ) ) {
            return new WP_Error( 'fliz_invalid_video_id', __( 'Invalid video ID format.', 'fliz-video-generator' ) );
		}

		$result = $this->request( 'GET', '/api/rest/videos/' . $video_id );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		if ( isset( $result['fliz_video_by_pk'] ) ) {
			return $result['fliz_video_by_pk'];
		}

        return new WP_Error( 'fliz_video_not_found', __( 'Video not found.', 'fliz-video-generator' ) );
	}

	/**
     * Vérifier le statut d'une vidéo
	 *
     * Raccourci vers get_video() qui retourne le step/status
	 *
	 * @param string $video_id UUID de la vidéo.
	 * @return array|WP_Error Contient step, url, error.
	 */
	public function get_video_status( $video_id ) {
		$video = $this->get_video( $video_id );

		if ( is_wp_error( $video ) ) {
			return $video;
		}

		return array(
			'video_id' => $video['id'],
			'status'   => $video['step'],
            'url'      => isset( $video['url'] ) ? $video['url'] : null,
            'error'    => isset( $video['error'] ) ? $video['error'] : null,
            'title'    => isset( $video['title'] ) ? $video['title'] : '',
		);
	}

	/**
     * Lister les vidéos
	 *
	 * GET /api/rest/videos
	 *
	 * @param int $limit  Nombre de résultats (défaut: 20).
	 * @param int $offset Décalage pour pagination (défaut: 0).
	 * @return array|WP_Error
	 */
	public function get_videos( $limit = 20, $offset = 0 ) {
		$endpoint = sprintf(
			'/api/rest/videos?limit=%d&offset=%d',
			absint( $limit ),
			absint( $offset )
		);

		$result = $this->request( 'GET', $endpoint );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

        return isset( $result['fliz_video'] ) ? $result['fliz_video'] : array();
	}

	/**
     * Récupérer les voix disponibles
	 *
	 * GET /api/rest/voices
	 *
	 * @return array|WP_Error Liste des voix.
	 */
	public function get_voices() {
		$result = $this->request( 'GET', '/api/rest/voices' );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

        return isset( $result['fliz_list_voices']['voices'] ) ? $result['fliz_list_voices']['voices'] : array();
	}

	/**
     * Récupérer les musiques disponibles
	 *
	 * GET /api/rest/musics
	 *
	 * @return array|WP_Error Liste des musiques par thème.
	 */
	public function get_musics() {
		$result = $this->request( 'GET', '/api/rest/musics' );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

        return isset( $result['fliz_list_musics']['musics'] ) ? $result['fliz_list_musics']['musics'] : array();
	}

	/**
     * Traduire une vidéo existante
	 *
	 * POST /api/rest/videos/{from_video_id}/translate
	 *
	 * @param string $video_id ID de la vidéo source.
	 * @param string $new_lang Nouvelle langue (code ISO 639-1).
	 * @return array|WP_Error Contient new_video_id et cost.
	 */
	public function translate_video( $video_id, $new_lang ) {
		$video_id = sanitize_text_field( $video_id );
		$new_lang = sanitize_text_field( $new_lang );

		$endpoint = '/api/rest/videos/' . $video_id . '/translate?new_lang=' . $new_lang;

		$body = array(
			'from_video_id' => $video_id,
			'new_lang'      => $new_lang,
		);

		$result = $this->request( 'POST', $endpoint, $body );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		if ( isset( $result['fliz_video_translate'] ) ) {
			return array(
				'new_video_id' => $result['fliz_video_translate']['new_video_id'],
				'cost'         => $result['fliz_video_translate']['cost'],
			);
		}

        return new WP_Error( 'fliz_invalid_response', __( 'Invalid API response.', 'fliz-video-generator' ) );
	}

	/**
     * Dupliquer une vidéo existante
	 *
	 * POST /api/rest/videos/{from_video_id}/duplicate
	 *
	 * @param string $video_id ID de la vidéo source.
	 * @return array|WP_Error Contient new_video_id.
	 */
	public function duplicate_video( $video_id ) {
		$video_id = sanitize_text_field( $video_id );

		$endpoint = '/api/rest/videos/' . $video_id . '/duplicate';

		$body = array(
			'from_video_id' => $video_id,
		);

		$result = $this->request( 'POST', $endpoint, $body );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		if ( isset( $result['fliz_video_duplicate'] ) ) {
			return array(
				'new_video_id' => $result['fliz_video_duplicate']['new_video_id'],
			);
		}

        return new WP_Error( 'fliz_invalid_response', __( 'Invalid API response.', 'fliz-video-generator' ) );
	}

	/**
     * Vérifier si une vidéo est prête
	 *
	 * @param string $video_id UUID de la vidéo.
	 * @return bool
	 */
	public function is_video_ready( $video_id ) {
		$status = $this->get_video_status( $video_id );

		if ( is_wp_error( $status ) ) {
			return false;
		}

		return 'complete' === $status['status'];
	}

	/**
     * Vérifier si une vidéo a échoué
	 *
	 * @param string $video_id UUID de la vidéo.
	 * @return bool
	 */
	public function is_video_failed( $video_id ) {
		$status = $this->get_video_status( $video_id );

		if ( is_wp_error( $status ) ) {
			return true;
		}

		return in_array( $status['status'], array( 'failed', 'failed_unrecoverable', 'block' ), true );
	}

	/**
     * Obtenir les énumérations statiques
	 *
	 * @return array Toutes les options disponibles.
	 */
	public static function get_enums() {
		return array(
			'formats'           => array(
				'size_16_9' => __( 'Landscape 16:9 (YouTube)', 'fliz-video-generator' ),
				'size_9_16' => __( 'Portrait 9:16 (TikTok, Reels)', 'fliz-video-generator' ),
				'square'    => __( 'Square 1:1 (Instagram)', 'fliz-video-generator' ),
			),
			'categories'        => array(
				'article' => __( 'Article', 'fliz-video-generator' ),
				'product' => __( 'Product', 'fliz-video-generator' ),
				'ad'      => __( 'Advertisement', 'fliz-video-generator' ),
			),
			'script_styles'     => array(
				'classified_ad'                     => __( 'Classified Ad', 'fliz-video-generator' ),
				'cooking_recipe'                    => __( 'Cooking Recipe', 'fliz-video-generator' ),
				'ecommerce_commercial_presentation' => __( 'E-commerce Presentation', 'fliz-video-generator' ),
				'ecommerce_product_test'            => __( 'Product Test', 'fliz-video-generator' ),
				'news_narrative_podcast_style'      => __( 'Podcast Style', 'fliz-video-generator' ),
				'news_social_media_style'           => __( 'Social Media Style', 'fliz-video-generator' ),
				'news_tv_style'                     => __( 'TV News Style', 'fliz-video-generator' ),
				'news_youtube_educational_style'    => __( 'YouTube Educational', 'fliz-video-generator' ),
				'real_estate_listing'               => __( 'Real Estate Listing', 'fliz-video-generator' ),
				'step_by_step_tutorial'             => __( 'Step-by-Step Tutorial', 'fliz-video-generator' ),
			),
			'caption_styles'    => array(
				'animated_background' => __( 'Animated Background', 'fliz-video-generator' ),
				'bouncing_background' => __( 'Bouncing Background', 'fliz-video-generator' ),
				'colored_words'       => __( 'Colored Words', 'fliz-video-generator' ),
				'scaling_words'       => __( 'Scaling Words', 'fliz-video-generator' ),
			),
			'caption_positions' => array(
				'bottom' => __( 'Bottom', 'fliz-video-generator' ),
				'center' => __( 'Center', 'fliz-video-generator' ),
			),
			'caption_fonts'     => array(
				'cabin'      => 'Cabin',
				'inter'      => 'Inter',
				'lato'       => 'Lato',
				'montserrat' => 'Montserrat',
				'nunito'     => 'Nunito',
				'open_sans'  => 'Open Sans',
				'oswald'     => 'Oswald',
				'poppins'    => 'Poppins',
				'roboto'     => 'Roboto',
				'ubuntu'     => 'Ubuntu',
			),
			'image_styles'      => array(
				'hyperrealistic'              => __( 'Hyperrealistic', 'fliz-video-generator' ),
				'digital_art'                 => __( 'Digital Art', 'fliz-video-generator' ),
				'african_pattern'             => __( 'African Pattern', 'fliz-video-generator' ),
				'australian_aboriginal_art'   => __( 'Australian Aboriginal Art', 'fliz-video-generator' ),
				'brazilian_cordel_art'        => __( 'Brazilian Cordel Art', 'fliz-video-generator' ),
				'charcoal'                    => __( 'Charcoal', 'fliz-video-generator' ),
				'classic_american_animation'  => __( 'Classic American Animation', 'fliz-video-generator' ),
				'classic_cartoon_1'           => __( 'Classic Cartoon 1', 'fliz-video-generator' ),
				'classic_cartoon_2'           => __( 'Classic Cartoon 2', 'fliz-video-generator' ),
				'classical_painting'          => __( 'Classical Painting', 'fliz-video-generator' ),
				'clay_animation'              => __( 'Clay Animation', 'fliz-video-generator' ),
				'color_pencil'                => __( 'Color Pencil', 'fliz-video-generator' ),
				'comic_book'                  => __( 'Comic Book', 'fliz-video-generator' ),
				'creative_collage'            => __( 'Creative Collage', 'fliz-video-generator' ),
				'cut_out_sticker'             => __( 'Cut Out Sticker', 'fliz-video-generator' ),
				'cuteyukimix'                 => __( 'Cuteyukimix', 'fliz-video-generator' ),
				'cyber_fantasy'               => __( 'Cyber Fantasy', 'fliz-video-generator' ),
				'dark_cinematography'         => __( 'Dark Cinematography', 'fliz-video-generator' ),
				'dark_digital'                => __( 'Dark Digital', 'fliz-video-generator' ),
				'dark_fantaisy'               => __( 'Dark Fantasy', 'fliz-video-generator' ),
				'experimental_photography'    => __( 'Experimental Photography', 'fliz-video-generator' ),
				'fantasy'                     => __( 'Fantasy', 'fliz-video-generator' ),
				'fine_art_monochrome'         => __( 'Fine Art Monochrome', 'fliz-video-generator' ),
				'future_vision'               => __( 'Future Vision', 'fliz-video-generator' ),
				'futurism'                    => __( 'Futurism', 'fliz-video-generator' ),
				'horror'                      => __( 'Horror', 'fliz-video-generator' ),
				'indigenous_amazonian_art'    => __( 'Indigenous Amazonian Art', 'fliz-video-generator' ),
				'ink_art'                     => __( 'Ink Art', 'fliz-video-generator' ),
				'japan_sumi_e_ink'            => __( 'Japan Sumi-e Ink', 'fliz-video-generator' ),
				'japanese_animation'          => __( 'Japanese Animation', 'fliz-video-generator' ),
				'japanese_horror'             => __( 'Japanese Horror', 'fliz-video-generator' ),
				'japanese_woodblock_print'    => __( 'Japanese Woodblock Print', 'fliz-video-generator' ),
				'kawaii'                      => __( 'Kawaii', 'fliz-video-generator' ),
				'manga'                       => __( 'Manga', 'fliz-video-generator' ),
				'marker_art'                  => __( 'Marker Art', 'fliz-video-generator' ),
				'mexican_painting'            => __( 'Mexican Painting', 'fliz-video-generator' ),
				'minimalist'                  => __( 'Minimalist', 'fliz-video-generator' ),
				'modern_gaming'               => __( 'Modern Gaming', 'fliz-video-generator' ),
				'neo_pop'                     => __( 'Neo Pop', 'fliz-video-generator' ),
				'neon_glow'                   => __( 'Neon Glow', 'fliz-video-generator' ),
				'persian_art'                 => __( 'Persian Art', 'fliz-video-generator' ),
				'pixel_art'                   => __( 'Pixel Art', 'fliz-video-generator' ),
				'pop_art_1'                   => __( 'Pop Art 1', 'fliz-video-generator' ),
				'pop_art_2'                   => __( 'Pop Art 2', 'fliz-video-generator' ),
				'propaganda_style'            => __( 'Propaganda Style', 'fliz-video-generator' ),
				'psychedelic_vision'          => __( 'Psychedelic Vision', 'fliz-video-generator' ),
				'retro_gaming'                => __( 'Retro Gaming', 'fliz-video-generator' ),
				'retro_nostalgia'             => __( 'Retro Nostalgia', 'fliz-video-generator' ),
				'seventies_vibe'              => __( 'Seventies Vibe', 'fliz-video-generator' ),
				'silhouette_art'              => __( 'Silhouette Art', 'fliz-video-generator' ),
				'stick_figure_drawing'        => __( 'Stick Figure Drawing', 'fliz-video-generator' ),
				'storybook_1'                 => __( 'Storybook 1', 'fliz-video-generator' ),
				'storybook_2'                 => __( 'Storybook 2', 'fliz-video-generator' ),
				'storybook_3'                 => __( 'Storybook 3', 'fliz-video-generator' ),
				'style_3d_animation'          => __( '3D Animation', 'fliz-video-generator' ),
				'tarot_cards'                 => __( 'Tarot Cards', 'fliz-video-generator' ),
				'traditional_indian'          => __( 'Traditional Indian', 'fliz-video-generator' ),
				'vintage_decay'               => __( 'Vintage Decay', 'fliz-video-generator' ),
				'vintage_illustration'        => __( 'Vintage Illustration', 'fliz-video-generator' ),
				'vintage_tattoo'              => __( 'Vintage Tattoo', 'fliz-video-generator' ),
				'watercolor'                  => __( 'Watercolor', 'fliz-video-generator' ),
				'wool_felt_art'               => __( 'Wool Felt Art', 'fliz-video-generator' ),
			),
			'languages'         => array(
				'en' => __( 'English', 'fliz-video-generator' ),
				'fr' => __( 'French', 'fliz-video-generator' ),
				'es' => __( 'Spanish', 'fliz-video-generator' ),
				'de' => __( 'German', 'fliz-video-generator' ),
				'it' => __( 'Italian', 'fliz-video-generator' ),
				'pt' => __( 'Portuguese', 'fliz-video-generator' ),
				'nl' => __( 'Dutch', 'fliz-video-generator' ),
				'pl' => __( 'Polish', 'fliz-video-generator' ),
				'ru' => __( 'Russian', 'fliz-video-generator' ),
				'ja' => __( 'Japanese', 'fliz-video-generator' ),
				'ko' => __( 'Korean', 'fliz-video-generator' ),
				'zh' => __( 'Chinese', 'fliz-video-generator' ),
				'ar' => __( 'Arabic', 'fliz-video-generator' ),
				'hi' => __( 'Hindi', 'fliz-video-generator' ),
				'tr' => __( 'Turkish', 'fliz-video-generator' ),
				'sv' => __( 'Swedish', 'fliz-video-generator' ),
				'da' => __( 'Danish', 'fliz-video-generator' ),
				'fi' => __( 'Finnish', 'fliz-video-generator' ),
				'no' => __( 'Norwegian', 'fliz-video-generator' ),
				'cs' => __( 'Czech', 'fliz-video-generator' ),
                'ro' => __( 'Romanian', 'fliz-video-generator' ),
                'hu' => __( 'Hungarian', 'fliz-video-generator' ),
				'el' => __( 'Greek', 'fliz-video-generator' ),
				'he' => __( 'Hebrew', 'fliz-video-generator' ),
				'th' => __( 'Thai', 'fliz-video-generator' ),
				'vi' => __( 'Vietnamese', 'fliz-video-generator' ),
                'id' => __( 'Indonesian', 'fliz-video-generator' ),
			),
		);
	}
}
