<?php
/**
 * Template d'affichage de la metabox
 *
 * @package Fliz_Video_Generator
 *
 * Variables disponibles:
 * @var WP_Post $post      Article courant
 * @var string  $video_id  ID de la vidéo Fliz
 * @var string  $video_url URL de la vidéo
 * @var string  $status    Statut de la vidéo (pending, complete, error)
 */

// Empêcher l'accès direct
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

$fliz_options = get_option( 'fliz_settings', array() );
$fliz_enums   = Fliz_API_Client::get_enums();

// Déterminer l'état actuel
$fliz_has_video  = ! empty( $video_id );
$fliz_is_pending = 'pending' === $status;
$fliz_is_error   = 'error' === $status;
$fliz_is_ready   = 'complete' === $status && ! empty( $video_url );
?>

<div class="fliz-metabox-content" data-post-id="<?php echo esc_attr( $post->ID ); ?>" data-video-id="<?php echo esc_attr( $video_id ); ?>">
    
    <?php if ( $fliz_is_ready ) : ?>
        <!-- État: Vidéo prête -->
        <div class="fliz-video-ready">
            <div class="fliz-video-preview">
                <video controls preload="metadata">
                    <source src="<?php echo esc_url( $video_url ); ?>" type="video/mp4">
                </video>
            </div>

            <div class="fliz-shortcode-display">
                <code>[fliz_video id="<?php echo esc_attr( $video_id ); ?>"]</code>
                <button type="button" class="button button-small fliz-copy-shortcode">
                    <span class="dashicons dashicons-clipboard"></span>
                </button>
            </div>

            <div class="fliz-video-actions">
                <button type="button" class="button" id="fliz-regenerate-btn">
                    <span class="dashicons dashicons-update"></span>
                    <?php esc_html_e( 'Regenerate', 'fliz-video-generator' ); ?>
                </button>
            </div>
        </div>

    <?php elseif ( $fliz_is_pending ) : ?>
        <!-- État: En cours de génération -->
        <div class="fliz-loading-state" data-video-id="<?php echo esc_attr( $video_id ); ?>">
            <div class="fliz-spinner"></div>
            <p><?php esc_html_e( 'Generating video...', 'fliz-video-generator' ); ?></p>
            <p class="fliz-status-text"><?php esc_html_e( 'This may take 2 to 10 minutes.', 'fliz-video-generator' ); ?></p>
        </div>

    <?php elseif ( $fliz_is_error ) : ?>
        <!-- État: Erreur -->
        <div class="fliz-error-state">
            <span class="dashicons dashicons-warning"></span>
            <p><?php esc_html_e( 'Video generation failed.', 'fliz-video-generator' ); ?></p>
            <button type="button" class="button button-primary" id="fliz-retry-btn">
                <?php esc_html_e( 'Retry', 'fliz-video-generator' ); ?>
            </button>
        </div>

    <?php else : ?>
        <!-- État: Pas de vidéo - Formulaire de génération -->
        <div class="fliz-generate-form">
            <!-- Sélecteur de format -->
            <div class="fliz-field">
                <label class="fliz-field-label"><?php esc_html_e( 'Format', 'fliz-video-generator' ); ?></label>
                <div class="fliz-format-selector">
                    <?php
                    $fliz_default_format = isset( $fliz_options['default_format'] ) ? $fliz_options['default_format'] : 'size_16_9';
                    foreach ( $fliz_enums['formats'] as $fliz_format_value => $fliz_format_label ) :
                        $fliz_format_icon = 'size_16_9' === $fliz_format_value ? '▬' : ( 'size_9_16' === $fliz_format_value ? '▮' : '■' );
                        ?>
                        <label class="fliz-format-option">
                            <input 
                                type="radio" 
                                name="fliz_format" 
                                value="<?php echo esc_attr( $fliz_format_value ); ?>"
                                <?php checked( $fliz_default_format, $fliz_format_value ); ?>
                            />
                            <span class="fliz-format-btn">
                                <span class="fliz-format-icon"><?php echo esc_html( $fliz_format_icon ); ?></span>
                                <span class="fliz-format-text"><?php echo esc_html( str_replace( array( 'Landscape ', 'Portrait ', 'Square ' ), '', $fliz_format_label ) ); ?></span>
                            </span>
                        </label>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Checkbox auto-generate -->
            <div class="fliz-field fliz-checkbox-field">
                <label>
                    <input 
                        type="checkbox" 
                        name="fliz_auto_generate" 
                        value="1"
                        <?php checked( get_post_meta( $post->ID, '_fliz_auto_generate', true ) ); ?>
                    />
                    <?php esc_html_e( 'Auto-generate on publish', 'fliz-video-generator' ); ?>
                </label>
            </div>

            <!-- Bouton de génération -->
            <button type="button" class="button button-primary button-large fliz-generate-btn" id="fliz-generate-btn">
                <span class="dashicons dashicons-video-alt3"></span>
                <?php esc_html_e( 'Generate Video', 'fliz-video-generator' ); ?>
            </button>
        </div>
    <?php endif; ?>

</div>
