<?php
/**
 * Template de la page des logs
 *
 * @package Fliz_Video_Generator
 *
 * Variables disponibles:
 * @var array  $logs          Liste des logs
 * @var array  $counts        Compteurs par statut
 * @var string $status_filter Filtre de statut actuel
 * @var int    $page          Page courante
 * @var int    $per_page      Éléments par page
 */

// Empêcher l'accès direct
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Calculer la pagination
$fliz_total_items = $counts['all'];
$fliz_total_pages = ceil( $fliz_total_items / $per_page );
?>
<div class="wrap fliz-logs-wrap">
    <h1 class="fliz-page-title">
        <span class="fliz-logo-icon">
            <?php Fliz_Settings::echo_menu_icon_svg(); ?>
        </span>
        <?php esc_html_e( 'Video Generation Logs', 'fliz-video-generator' ); ?>
    </h1>

    <?php wp_nonce_field( 'fliz_logs_nonce', 'fliz_logs_nonce_field' ); ?>

    <!-- Filtres de statut -->
    <ul class="subsubsub fliz-status-filters">
        <li>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=fliz-logs' ) ); ?>" 
               class="<?php echo esc_attr( 'all' === $status_filter ? 'current' : '' ); ?>">
                <?php esc_html_e( 'All', 'fliz-video-generator' ); ?>
                <span class="count">(<?php echo esc_html( $counts['all'] ); ?>)</span>
            </a> |
        </li>
        <li>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=fliz-logs&status=pending' ) ); ?>"
               class="<?php echo esc_attr( 'pending' === $status_filter ? 'current' : '' ); ?>">
                <?php esc_html_e( 'Pending', 'fliz-video-generator' ); ?>
                <span class="count">(<?php echo esc_html( $counts['pending'] ); ?>)</span>
            </a> |
        </li>
        <li>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=fliz-logs&status=complete' ) ); ?>"
               class="<?php echo esc_attr( 'complete' === $status_filter ? 'current' : '' ); ?>">
                <?php esc_html_e( 'Complete', 'fliz-video-generator' ); ?>
                <span class="count">(<?php echo esc_html( $counts['complete'] ); ?>)</span>
            </a> |
        </li>
        <li>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=fliz-logs&status=error' ) ); ?>"
               class="<?php echo esc_attr( 'error' === $status_filter ? 'current' : '' ); ?>">
                <?php esc_html_e( 'Error', 'fliz-video-generator' ); ?>
                <span class="count">(<?php echo esc_html( $counts['error'] ); ?>)</span>
            </a>
        </li>
    </ul>

    <!-- Tableau des logs -->
    <table class="wp-list-table widefat fixed striped fliz-logs-table">
        <thead>
            <tr>
                <th class="manage-column column-date"><?php esc_html_e( 'Date', 'fliz-video-generator' ); ?></th>
                <th class="manage-column column-post"><?php esc_html_e( 'Post', 'fliz-video-generator' ); ?></th>
                <th class="manage-column column-video-id"><?php esc_html_e( 'Video ID', 'fliz-video-generator' ); ?></th>
                <th class="manage-column column-status"><?php esc_html_e( 'Status', 'fliz-video-generator' ); ?></th>
                <th class="manage-column column-message"><?php esc_html_e( 'Message', 'fliz-video-generator' ); ?></th>
                <th class="manage-column column-actions"><?php esc_html_e( 'Actions', 'fliz-video-generator' ); ?></th>
            </tr>
        </thead>
        <tbody>
            <?php if ( empty( $logs ) ) : ?>
                <tr>
                    <td colspan="6" class="fliz-no-logs">
                        <?php esc_html_e( 'No logs found.', 'fliz-video-generator' ); ?>
                    </td>
                </tr>
            <?php else : ?>
                <?php foreach ( $logs as $fliz_log ) : ?>
                    <?php
                    $fliz_post_title = get_the_title( $fliz_log->post_id );
                    $fliz_edit_url   = get_edit_post_link( $fliz_log->post_id, 'raw' );
                    
                    // Classe de statut
                    $fliz_status_class = 'fliz-status-' . esc_attr( $fliz_log->status );
                    
                    // Label de statut
                    $fliz_status_labels = array(
                        'pending'  => __( 'Pending', 'fliz-video-generator' ),
                        'complete' => __( 'Complete', 'fliz-video-generator' ),
                        'error'    => __( 'Error', 'fliz-video-generator' ),
                    );
                    $fliz_status_label = isset( $fliz_status_labels[ $fliz_log->status ] ) ? $fliz_status_labels[ $fliz_log->status ] : $fliz_log->status;
                    ?>
                    <tr data-log-id="<?php echo esc_attr( $fliz_log->id ); ?>" data-video-id="<?php echo esc_attr( $fliz_log->video_id ); ?>">
                        <td class="column-date">
                            <?php echo esc_html( mysql2date( 'Y-m-d H:i:s', $fliz_log->created_at ) ); ?>
                        </td>
                        <td class="column-post">
                            <?php if ( $fliz_edit_url ) : ?>
                                <a href="<?php echo esc_url( $fliz_edit_url ); ?>">
                                    <?php echo esc_html( $fliz_post_title ? $fliz_post_title : __( '(no title)', 'fliz-video-generator' ) ); ?>
                                </a>
                            <?php else : ?>
                                <?php echo esc_html( $fliz_post_title ? $fliz_post_title : __( '(deleted)', 'fliz-video-generator' ) ); ?>
                            <?php endif; ?>
                        </td>
                        <td class="column-video-id">
                            <code><?php echo esc_html( substr( $fliz_log->video_id, 0, 8 ) . '...' ); ?></code>
                        </td>
                        <td class="column-status">
                            <span class="fliz-status-badge <?php echo esc_attr( $fliz_status_class ); ?>">
                                <?php echo esc_html( $fliz_status_label ); ?>
                            </span>
                        </td>
                        <td class="column-message">
                            <?php echo esc_html( $fliz_log->message ); ?>
                        </td>
                        <td class="column-actions">
                            <?php if ( 'pending' === $fliz_log->status ) : ?>
                                <button type="button" class="button button-small fliz-refresh-status" title="<?php esc_attr_e( 'Refresh Status', 'fliz-video-generator' ); ?>">
                                    <span class="dashicons dashicons-update"></span>
                                </button>
                            <?php elseif ( 'complete' === $fliz_log->status ) : ?>
                                <?php
                                $fliz_video_url = get_post_meta( $fliz_log->post_id, '_fliz_video_url', true );
                                if ( $fliz_video_url ) :
                                    ?>
                                    <a href="<?php echo esc_url( $fliz_video_url ); ?>" target="_blank" class="button button-small" title="<?php esc_attr_e( 'View Video', 'fliz-video-generator' ); ?>">
                                        <span class="dashicons dashicons-visibility"></span>
                                    </a>
                                <?php endif; ?>
                            <?php elseif ( 'error' === $fliz_log->status ) : ?>
                                <button type="button" class="button button-small fliz-retry-generation" title="<?php esc_attr_e( 'Retry', 'fliz-video-generator' ); ?>">
                                    <span class="dashicons dashicons-controls-repeat"></span>
                                </button>
                            <?php endif; ?>
                            <button type="button" class="button button-small fliz-delete-log" title="<?php esc_attr_e( 'Delete Log', 'fliz-video-generator' ); ?>">
                                <span class="dashicons dashicons-trash"></span>
                            </button>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>

    <!-- Pagination -->
    <?php if ( $fliz_total_pages > 1 ) : ?>
        <div class="tablenav bottom">
            <div class="tablenav-pages">
                <span class="displaying-num">
                    <?php
                    printf(
                        /* translators: %s: number of items */
                        esc_html( _n( '%s item', '%s items', $fliz_total_items, 'fliz-video-generator' ) ),
                        esc_html( number_format_i18n( $fliz_total_items ) )
                    );
                    ?>
                </span>
                <span class="pagination-links">
                    <?php
                    $fliz_base_url = admin_url( 'admin.php?page=fliz-logs' );
                    if ( 'all' !== $status_filter ) {
                        $fliz_base_url = add_query_arg( 'status', $status_filter, $fliz_base_url );
                    }

                    // Première page
                    if ( $page > 1 ) :
                        ?>
                        <a class="first-page button" href="<?php echo esc_url( $fliz_base_url ); ?>">
                            <span class="screen-reader-text"><?php esc_html_e( 'First page', 'fliz-video-generator' ); ?></span>
                            <span aria-hidden="true">«</span>
                        </a>
                        <a class="prev-page button" href="<?php echo esc_url( add_query_arg( 'paged', $page - 1, $fliz_base_url ) ); ?>">
                            <span class="screen-reader-text"><?php esc_html_e( 'Previous page', 'fliz-video-generator' ); ?></span>
                            <span aria-hidden="true">‹</span>
                        </a>
                    <?php else : ?>
                        <span class="tablenav-pages-navspan button disabled" aria-hidden="true">«</span>
                        <span class="tablenav-pages-navspan button disabled" aria-hidden="true">‹</span>
                    <?php endif; ?>

                    <span class="paging-input">
                        <span class="tablenav-paging-text">
                            <?php echo esc_html( $page ); ?> <?php esc_html_e( 'of', 'fliz-video-generator' ); ?> <span class="total-pages"><?php echo esc_html( $fliz_total_pages ); ?></span>
                        </span>
                    </span>

                    <?php if ( $page < $fliz_total_pages ) : ?>
                        <a class="next-page button" href="<?php echo esc_url( add_query_arg( 'paged', $page + 1, $fliz_base_url ) ); ?>">
                            <span class="screen-reader-text"><?php esc_html_e( 'Next page', 'fliz-video-generator' ); ?></span>
                            <span aria-hidden="true">›</span>
                        </a>
                        <a class="last-page button" href="<?php echo esc_url( add_query_arg( 'paged', $fliz_total_pages, $fliz_base_url ) ); ?>">
                            <span class="screen-reader-text"><?php esc_html_e( 'Last page', 'fliz-video-generator' ); ?></span>
                            <span aria-hidden="true">»</span>
                        </a>
                    <?php else : ?>
                        <span class="tablenav-pages-navspan button disabled" aria-hidden="true">›</span>
                        <span class="tablenav-pages-navspan button disabled" aria-hidden="true">»</span>
                    <?php endif; ?>
                </span>
            </div>
        </div>
    <?php endif; ?>
</div>

<script>
jQuery(document).ready(function($) {
    var flizNonce = $('#fliz_logs_nonce_field').val();

    // Rafraîchir le statut
    $('.fliz-refresh-status').on('click', function() {
        var $btn = $(this);
        var $row = $btn.closest('tr');
        var flizVideoId = $row.data('video-id');

        $btn.prop('disabled', true).find('.dashicons').addClass('spin');

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'fliz_refresh_log_status',
                security: flizNonce,
                video_id: flizVideoId
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                }
            },
            complete: function() {
                $btn.prop('disabled', false).find('.dashicons').removeClass('spin');
            }
        });
    });

    // Supprimer un log
    $('.fliz-delete-log').on('click', function() {
        if (!confirm('<?php echo esc_js( __( 'Are you sure you want to delete this log?', 'fliz-video-generator' ) ); ?>')) {
            return;
        }

        var $btn = $(this);
        var $row = $btn.closest('tr');
        var flizLogId = $row.data('log-id');

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'fliz_delete_log',
                security: flizNonce,
                log_id: flizLogId
            },
            success: function(response) {
                if (response.success) {
                    $row.fadeOut(300, function() {
                        $(this).remove();
                    });
                }
            }
        });
    });
});
</script>
