<?php
/**
 * Classe Fliz_Settings
 *
 * Gère la page de réglages du plugin.
 *
 * @package Fliz_Video_Generator
 */

// Empêcher l'accès direct
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Classe Fliz_Settings
 */
class Fliz_Settings {

    /**
     * Nom de l'option
     */
    private const OPTION_NAME = 'fliz_settings';

    /**
     * Groupe d'options
     */
    private const OPTION_GROUP = 'fliz_options_group';

    /**
     * Page de réglages
     */
    private const PAGE_SLUG = 'fliz-settings';

    /**
     * Initialiser les réglages
     */
    public function init() {
        add_action( 'admin_menu', array( $this, 'add_menu_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'wp_ajax_fliz_test_api', array( $this, 'ajax_test_api' ) );
        add_action( 'wp_ajax_fliz_refresh_options', array( $this, 'ajax_refresh_options' ) );
    }

    /**
     * Ajouter la page de menu
     */
    public function add_menu_page() {
        add_menu_page(
            __( 'Fliz Video', 'fliz-video-generator' ),
            __( 'Fliz Video', 'fliz-video-generator' ),
            'manage_options',
            self::PAGE_SLUG,
            array( $this, 'render_settings_page' ),
            'data:image/svg+xml;base64,' . base64_encode( self::get_menu_icon_svg() ),
            100
        );

        add_submenu_page(
            self::PAGE_SLUG,
            __( 'Settings', 'fliz-video-generator' ),
            __( 'Settings', 'fliz-video-generator' ),
            'manage_options',
            self::PAGE_SLUG,
            array( $this, 'render_settings_page' )
        );
    }

    /**
     * Récupérer le SVG de l'icône du menu
     *
     * @return string
     */
    public static function get_menu_icon_svg() {
        return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 309 346" fill="currentColor"><path d="M144.54,285.17l10.1-28.96,15.16-43.46c.05-.16.09-.3.11-.48v-.02c.27-1.28-.16-2.57-1-3.48-.71-.77-1.73-1.25-2.91-1.25h-53.85c-3.05,0-4.98-3.23-3.53-5.9l20.51-37.35,1.37-2.5,4.16-7.6h.02l7.47-13.64c.71-1.34.59-2.8-.11-3.94-.68-1.14-1.93-1.94-3.42-1.94h-52.63c-3.03,0-4.96-3.23-3.53-5.9l58.95-109.68C62.21,25.7,0,92.08,0,173s56.54,141.07,130.43,152.63l14.11-40.46Z"/><path d="M202.75,26.2l-35.9,68.39c-1.43,2.68.5,5.9,3.53,5.9h52.63c3.03,0,4.96,3.25,3.53,5.92l-32.53,60.64-2.19,4.1-4.17,7.78c-.18.34-.32.68-.39,1.03-.54,2.39,1.27,4.87,3.92,4.87h19.6c3.03,0,4.98,3.23,3.53,5.89l-27.77,51.63-13.41,24.93-10.88,20.24-21.19,39.38c4.43.38,8.91.59,13.44.59,85.33,0,154.5-69.17,154.5-154.5,0-68.48-44.55-126.54-106.25-146.8Z"/><path d="M209.03,3c2.69,0,4.4,2.85,3.16,5.23l-45.34,86.36c-1.43,2.68.5,5.9,3.53,5.9h52.63c3.03,0,4.96,3.25,3.53,5.92l-32.53,60.64-2.19,4.1-4.17,7.78c-.18.34-.32.68-.39,1.03-.54,2.39,1.27,4.87,3.92,4.87h19.6c3.03,0,4.98,3.23,3.53,5.89l-27.77,51.63-13.41,24.93-10.88,20.24-28.7,53.35c-.8,1.49-2.11,2.13-3.42,2.13-2.21,0-4.41-1.82-4.11-4.41.04-.27.09-.55.18-.84l18.33-52.58,10.1-28.96,15.16-43.46c.05-.16.09-.3.11-.48v-.02c.27-1.28-.16-2.57-1-3.48-.71-.77-1.73-1.25-2.91-1.25h-53.85c-3.05,0-4.98-3.23-3.53-5.9l20.51-37.35,1.37-2.5,4.16-7.6h.02l7.47-13.64c.71-1.34.59-2.8-.11-3.94-.68-1.14-1.93-1.94-3.42-1.94h-52.63c-3.03,0-4.96-3.23-3.53-5.9L149.03,4.87c.04-.05.07-.11.11-.16.64-1.07,1.8-1.71,3.05-1.71h56.84M209.03,0h-56.84c-2.27,0-4.4,1.18-5.58,3.1l-.05.08-.08.12-.09.16-66.58,123.87c-1.18,2.21-1.11,4.79.17,6.93,1.28,2.13,3.52,3.4,6.01,3.4h52.63c.51,0,.76.33.84.48.15.24.26.59.04,1l-6.6,12.06h0l-.87,1.56-4.16,7.6-1.37,2.49-20.51,37.35c-1.2,2.21-1.14,4.8.13,6.93,1.28,2.14,3.53,3.42,6.03,3.42h53.85c.29,0,.53.1.71.29.07.08.34.41.26.82l-.06.29-15.1,43.3-10.1,28.97-18.33,52.58-.02.04v.04c-.14.41-.24.85-.3,1.34-.24,2.02.36,3.91,1.67,5.38,1.35,1.52,3.38,2.43,5.41,2.43,2.55,0,4.81-1.39,6.06-3.71l28.7-53.35,10.88-20.24,13.41-24.93,27.77-51.63c1.18-2.17,1.13-4.74-.14-6.88-1.28-2.15-3.53-3.43-6.03-3.43h-19.6c-.32,0-.59-.12-.78-.37-.2-.25-.28-.56-.21-.84v-.03s.01-.03.01-.03c0,0,0,0,0,0,0,0,.02-.07.11-.23l4.16-7.75,2.2-4.11,32.53-60.64c1.18-2.21,1.11-4.8-.17-6.94-1.28-2.13-3.52-3.4-6-3.4h-52.63c-.5,0-.75-.3-.86-.49-.11-.18-.26-.56-.02-1.01L214.84,9.62c1.08-2.05,1.01-4.46-.19-6.45-1.2-1.98-3.3-3.17-5.62-3.17h0Z"/></svg>';
    }

    /**
     * Récupérer les balises HTML autorisées pour les SVG
     *
     * @return array
     */
    public static function get_svg_allowed_html() {
        return array(
            'svg'  => array(
                'xmlns'       => array(),
                'viewbox'     => array(),
                'fill'        => array(),
                'class'       => array(),
                'aria-hidden' => array(),
                'role'        => array(),
                'width'       => array(),
                'height'      => array(),
            ),
            'path' => array(
                'd'    => array(),
                'fill' => array(),
            ),
        );
    }

    /**
     * Afficher le SVG de l'icône du menu de manière sécurisée
     */
    public static function echo_menu_icon_svg() {
        echo wp_kses( self::get_menu_icon_svg(), self::get_svg_allowed_html() );
    }

    /**
     * Enregistrer les réglages
     */
    public function register_settings() {
        register_setting(
            self::OPTION_GROUP,
            self::OPTION_NAME,
            array(
                'type'              => 'array',
                'sanitize_callback' => array( $this, 'sanitize_settings' ),
                'default'           => $this->get_defaults(),
            )
        );

        // Section API
        add_settings_section(
            'fliz_api_section',
            __( 'API Configuration', 'fliz-video-generator' ),
            array( $this, 'render_api_section' ),
            self::PAGE_SLUG
        );

        add_settings_field(
            'api_key',
            __( 'API Key', 'fliz-video-generator' ),
            array( $this, 'render_api_key_field' ),
            self::PAGE_SLUG,
            'fliz_api_section'
        );

        // Section Défauts Vidéo
        add_settings_section(
            'fliz_defaults_section',
            __( 'Default Video Settings', 'fliz-video-generator' ),
            array( $this, 'render_defaults_section' ),
            self::PAGE_SLUG
        );

        $this->add_default_fields();
    }

    /**
     * Ajouter les champs par défaut
     */
    private function add_default_fields() {
        $fields = array(
            'default_format'            => __( 'Video Format', 'fliz-video-generator' ),
            'default_lang'              => __( 'Language', 'fliz-video-generator' ),
            'default_script_style'      => __( 'Script Style', 'fliz-video-generator' ),
            'default_image_style'       => __( 'Image Style', 'fliz-video-generator' ),
            'video_animation_mode'      => __( 'Animation Mode', 'fliz-video-generator' ),
            'default_voice_id'          => __( 'Default Voice', 'fliz-video-generator' ),
            'default_voice_gender'      => __( 'Voice Gender', 'fliz-video-generator' ),
            'default_music_id'          => __( 'Background Music', 'fliz-video-generator' ),
            'default_music_volume'      => __( 'Music Volume', 'fliz-video-generator' ),
            'default_caption_style'     => __( 'Caption Style', 'fliz-video-generator' ),
            'default_caption_position'  => __( 'Caption Position', 'fliz-video-generator' ),
            'default_caption_font'      => __( 'Caption Font', 'fliz-video-generator' ),
            'default_caption_color'     => __( 'Caption Color', 'fliz-video-generator' ),
            'default_caption_uppercase' => __( 'Force Uppercase', 'fliz-video-generator' ),
            'watermark_option'          => __( 'Watermark', 'fliz-video-generator' ),
            'auto_generate'             => __( 'Auto-generate on Publish', 'fliz-video-generator' ),
            'auto_insert_shortcode'     => __( 'Auto-insert Shortcode', 'fliz-video-generator' ),
            'is_automatic'              => __( 'Fully Automatic', 'fliz-video-generator' ),
        );

        foreach ( $fields as $field_id => $label ) {
            add_settings_field(
                $field_id,
                $label,
                array( $this, 'render_field_' . $field_id ),
                self::PAGE_SLUG,
                'fliz_defaults_section'
            );
        }
    }

    /**
     * Sanitizer les réglages
     *
     * @param array $input Données brutes.
     * @return array Données sanitisées.
     */
    public function sanitize_settings( $input ) {
        $sanitized = array();

        // API Key
        $sanitized['api_key'] = sanitize_text_field( $input['api_key'] ?? '' );

        // Format (enum Fliz)
        $valid_formats               = array( 'size_16_9', 'size_9_16', 'square' );
        $sanitized['default_format'] = in_array( $input['default_format'] ?? '', $valid_formats, true )
            ? $input['default_format']
            : 'size_16_9';

        // Catégorie (toujours "article")
        $sanitized['default_category'] = 'article';

        // Langue (code ISO 639-1)
        $sanitized['default_lang'] = sanitize_text_field( $input['default_lang'] ?? 'en' );

        // Style de script (enum Fliz)
        $valid_script_styles               = array(
            'classified_ad',
            'cooking_recipe',
            'ecommerce_commercial_presentation',
            'ecommerce_product_test',
            'news_narrative_podcast_style',
            'news_social_media_style',
            'news_tv_style',
            'news_youtube_educational_style',
            'real_estate_listing',
            'step_by_step_tutorial',
        );
        $sanitized['default_script_style'] = in_array( $input['default_script_style'] ?? '', $valid_script_styles, true )
            ? $input['default_script_style']
            : '';

        // Style d'image
        $sanitized['default_image_style'] = sanitize_text_field( $input['default_image_style'] ?? '' );

        // Voix
        $sanitized['default_voice_id'] = sanitize_text_field( $input['default_voice_id'] ?? '' );

        // Genre de voix
        $valid_voice_genders              = array( 'auto', 'male', 'female' );
        $sanitized['default_voice_gender'] = in_array( $input['default_voice_gender'] ?? '', $valid_voice_genders, true )
            ? $input['default_voice_gender']
            : 'auto';

        // Convertir voice_gender en is_male_voice pour l'API
        if ( 'male' === $sanitized['default_voice_gender'] ) {
            $sanitized['default_is_male_voice'] = true;
        } elseif ( 'female' === $sanitized['default_voice_gender'] ) {
            $sanitized['default_is_male_voice'] = false;
        } else {
            $sanitized['default_is_male_voice'] = null; // Auto
        }

        // Musique
        $sanitized['default_music_id']     = sanitize_text_field( $input['default_music_id'] ?? '' );
        $sanitized['default_music_volume'] = min( 100, max( 0, absint( $input['default_music_volume'] ?? 15 ) ) );

        // Sous-titres (captions)
        $valid_caption_styles               = array( 'animated_background', 'bouncing_background', 'colored_words', 'scaling_words' );
        $sanitized['default_caption_style'] = in_array( $input['default_caption_style'] ?? '', $valid_caption_styles, true )
            ? $input['default_caption_style']
            : 'animated_background';

        $valid_caption_positions               = array( 'bottom', 'center' );
        $sanitized['default_caption_position'] = in_array( $input['default_caption_position'] ?? '', $valid_caption_positions, true )
            ? $input['default_caption_position']
            : 'bottom';

        $valid_caption_fonts               = array( 'cabin', 'inter', 'lato', 'montserrat', 'nunito', 'open_sans', 'oswald', 'poppins', 'roboto', 'ubuntu' );
        $sanitized['default_caption_font'] = in_array( $input['default_caption_font'] ?? '', $valid_caption_fonts, true )
            ? $input['default_caption_font']
            : 'poppins';

        $sanitized['default_caption_color']     = sanitize_hex_color( $input['default_caption_color'] ?? '#FFFFFF' );
        $sanitized['default_caption_uppercase'] = ! empty( $input['default_caption_uppercase'] );

        // Video Animation Mode
        $valid_animation_modes              = array( '', 'hook_only', 'full_video' );
        $sanitized['video_animation_mode']  = in_array( $input['video_animation_mode'] ?? '', $valid_animation_modes, true )
            ? $input['video_animation_mode']
            : '';

        // Watermark
        $valid_watermark_types        = array( 'fliz', 'none', 'custom' );
        $sanitized['watermark_type']  = in_array( $input['watermark_type'] ?? '', $valid_watermark_types, true )
            ? $input['watermark_type']
            : 'fliz';
        $sanitized['watermark_url']   = esc_url_raw( $input['watermark_url'] ?? '' );

        // Si le type n'est pas "custom", on vide l'URL
        if ( 'custom' !== $sanitized['watermark_type'] ) {
            $sanitized['watermark_url'] = '';
        }

        // Automatisation
        $sanitized['auto_generate']         = ! empty( $input['auto_generate'] );
        $sanitized['auto_insert_shortcode'] = ! empty( $input['auto_insert_shortcode'] );
        $sanitized['is_automatic']          = ! empty( $input['is_automatic'] );

        return $sanitized;
    }

    /**
     * Valeurs par défaut
     *
     * @return array
     */
    private function get_defaults() {
        return array(
            'api_key'                   => '',
            'default_format'            => 'size_16_9',
            'default_category'          => 'article',
            'default_lang'              => 'en',
            'default_script_style'      => 'news_social_media_style',
            'default_image_style'       => 'hyperrealistic',
            'default_voice_id'          => '',
            'default_voice_gender'      => 'auto',
            'default_is_male_voice'     => null,
            'default_music_id'          => '',
            'default_music_volume'      => 15,
            'default_caption_style'     => 'animated_background',
            'default_caption_position'  => 'bottom',
            'default_caption_font'      => 'poppins',
            'default_caption_color'     => '#FFFFFF',
            'default_caption_uppercase' => false,
            'video_animation_mode'      => '',
            'watermark_type'            => 'fliz',
            'watermark_url'             => '',
            'auto_generate'             => false,
            'auto_insert_shortcode'     => true,
            'is_automatic'              => true,
        );
    }

    /**
     * Rendre la page de réglages
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        include FLIZ_PLUGIN_PATH . 'includes/admin/partials/settings-page.php';
    }

    /**
     * Rendre la section API
     */
    public function render_api_section() {
        echo '<p>' . esc_html__( 'Configure your Fliz API key to start generating videos.', 'fliz-video-generator' ) . '</p>';
        echo '<p>' . wp_kses(
            sprintf(
                /* translators: %s: URL du site Fliz */
                __( 'Get your API key at %s', 'fliz-video-generator' ),
                '<a href="https://app.fliz.ai/api-keys" target="_blank" rel="noopener noreferrer">https://app.fliz.ai/api-keys</a>'
            ),
            array(
                'a' => array(
                    'href'   => array(),
                    'target' => array(),
                    'rel'    => array(),
                ),
            )
        ) . '</p>';
    }

    /**
     * Rendre la section des défauts
     */
    public function render_defaults_section() {
        echo '<p>' . esc_html__( 'Set default values for video generation. These can be overridden per video.', 'fliz-video-generator' ) . '</p>';
    }

    /**
     * Rendre le champ API Key
     */
    public function render_api_key_field() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = isset( $options['api_key'] ) ? $options['api_key'] : '';
        ?>
        <div class="fliz-api-key-wrapper">
            <input 
                type="password" 
                id="fliz_api_key" 
                name="<?php echo esc_attr( self::OPTION_NAME ); ?>[api_key]" 
                value="<?php echo esc_attr( $value ); ?>" 
                class="regular-text"
                autocomplete="off"
            />
            <button type="button" class="button fliz-toggle-password">
                <span class="dashicons dashicons-visibility"></span>
            </button>
            <button type="button" class="button fliz-test-api" id="fliz-test-api-btn">
                <?php esc_html_e( 'Test Connection', 'fliz-video-generator' ); ?>
            </button>
            <span class="fliz-api-status" id="fliz-api-status"></span>
        </div>
        <p class="description">
            <?php esc_html_e( 'Your Fliz API key (JWT token).', 'fliz-video-generator' ); ?>
        </p>
        <?php
    }

    /**
     * Rendre le champ format
     */
    public function render_field_default_format() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = isset( $options['default_format'] ) ? $options['default_format'] : 'size_16_9';
        $formats = Fliz_API_Client::get_enums()['formats'];
        ?>
        <div class="fliz-format-selector fliz-format-selector-settings">
            <?php foreach ( $formats as $format_value => $format_label ) : ?>
                <label class="fliz-format-option">
                    <input 
                        type="radio" 
                        name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_format]" 
                        value="<?php echo esc_attr( $format_value ); ?>"
                        <?php checked( $value, $format_value ); ?>
                    />
                    <span class="fliz-format-btn"><?php echo esc_html( $format_label ); ?></span>
                </label>
            <?php endforeach; ?>
        </div>
        <?php
    }

    /**
     * Rendre le champ langue
     */
    public function render_field_default_lang() {
        $options   = get_option( self::OPTION_NAME, array() );
        $value     = isset( $options['default_lang'] ) ? $options['default_lang'] : 'en';
        $languages = Fliz_API_Client::get_enums()['languages'];
        ?>
        <select name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_lang]">
            <?php foreach ( $languages as $lang_code => $lang_name ) : ?>
                <option value="<?php echo esc_attr( $lang_code ); ?>" <?php selected( $value, $lang_code ); ?>>
                    <?php echo esc_html( $lang_name ); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php
    }

    /**
     * Rendre le champ style de script
     */
    public function render_field_default_script_style() {
        $options       = get_option( self::OPTION_NAME, array() );
        $value         = isset( $options['default_script_style'] ) ? $options['default_script_style'] : 'news_social_media_style';
        $script_styles = Fliz_API_Client::get_enums()['script_styles'];
        ?>
        <select name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_script_style]">
            <option value=""><?php esc_html_e( '— Auto —', 'fliz-video-generator' ); ?></option>
            <?php foreach ( $script_styles as $style_value => $style_label ) : ?>
                <option value="<?php echo esc_attr( $style_value ); ?>" <?php selected( $value, $style_value ); ?>>
                    <?php echo esc_html( $style_label ); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php
    }

    /**
     * Rendre le champ style d'image
     */
    public function render_field_default_image_style() {
        $options      = get_option( self::OPTION_NAME, array() );
        $value        = isset( $options['default_image_style'] ) ? $options['default_image_style'] : 'hyperrealistic';
        $image_styles = Fliz_API_Client::get_enums()['image_styles'];
        ?>
        <select name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_image_style]">
            <option value=""><?php esc_html_e( '— Auto —', 'fliz-video-generator' ); ?></option>
            <?php foreach ( $image_styles as $style_value => $style_label ) : ?>
                <option value="<?php echo esc_attr( $style_value ); ?>" <?php selected( $value, $style_value ); ?>>
                    <?php echo esc_html( $style_label ); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php
    }

    /**
     * Rendre le champ voix
     */
    public function render_field_default_voice_id() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = isset( $options['default_voice_id'] ) ? $options['default_voice_id'] : '';
        ?>
        <input 
            type="text" 
            name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_voice_id]" 
            value="<?php echo esc_attr( $value ); ?>" 
            class="regular-text"
            placeholder="<?php esc_attr_e( 'Auto-select voice', 'fliz-video-generator' ); ?>"
        />
        <p class="description">
            <?php esc_html_e( 'Leave empty to auto-select based on language. Voice IDs are available in the Fliz app.', 'fliz-video-generator' ); ?>
        </p>
        <?php
    }

    /**
     * Rendre le champ genre de voix
     */
    public function render_field_default_voice_gender() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = isset( $options['default_voice_gender'] ) ? $options['default_voice_gender'] : 'auto';
        ?>
        <select name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_voice_gender]">
            <option value="auto" <?php selected( $value, 'auto' ); ?>><?php esc_html_e( 'Auto', 'fliz-video-generator' ); ?></option>
            <option value="male" <?php selected( $value, 'male' ); ?>><?php esc_html_e( 'Male', 'fliz-video-generator' ); ?></option>
            <option value="female" <?php selected( $value, 'female' ); ?>><?php esc_html_e( 'Female', 'fliz-video-generator' ); ?></option>
        </select>
        <p class="description">
            <?php esc_html_e( 'Select the preferred voice gender for video narration.', 'fliz-video-generator' ); ?>
        </p>
        <?php
    }

    /**
     * Rendre le champ musique
     */
    public function render_field_default_music_id() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = isset( $options['default_music_id'] ) ? $options['default_music_id'] : '';
        ?>
        <input 
            type="text" 
            name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_music_id]" 
            value="<?php echo esc_attr( $value ); ?>" 
            class="regular-text"
            placeholder="<?php esc_attr_e( 'Auto-select music', 'fliz-video-generator' ); ?>"
        />
        <p class="description">
            <?php esc_html_e( 'Leave empty to auto-select music. Music IDs are available in the Fliz app.', 'fliz-video-generator' ); ?>
        </p>
        <?php
    }

    /**
     * Rendre le champ volume musique
     */
    public function render_field_default_music_volume() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = isset( $options['default_music_volume'] ) ? $options['default_music_volume'] : 15;
        ?>
        <input 
            type="range" 
            name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_music_volume]" 
            value="<?php echo esc_attr( $value ); ?>" 
            min="0"
            max="100"
            step="5"
            class="fliz-range-slider"
            id="fliz-music-volume"
        />
        <span class="fliz-range-value" id="fliz-music-volume-value"><?php echo esc_html( $value ); ?>%</span>
        <?php
    }

    /**
     * Rendre le champ style de caption
     */
    public function render_field_default_caption_style() {
        $options        = get_option( self::OPTION_NAME, array() );
        $value          = isset( $options['default_caption_style'] ) ? $options['default_caption_style'] : 'animated_background';
        $caption_styles = Fliz_API_Client::get_enums()['caption_styles'];
        ?>
        <select name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_caption_style]">
            <?php foreach ( $caption_styles as $style_value => $style_label ) : ?>
                <option value="<?php echo esc_attr( $style_value ); ?>" <?php selected( $value, $style_value ); ?>>
                    <?php echo esc_html( $style_label ); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php
    }

    /**
     * Rendre le champ position caption
     */
    public function render_field_default_caption_position() {
        $options           = get_option( self::OPTION_NAME, array() );
        $value             = isset( $options['default_caption_position'] ) ? $options['default_caption_position'] : 'bottom';
        $caption_positions = Fliz_API_Client::get_enums()['caption_positions'];
        ?>
        <div class="fliz-radio-group">
            <?php foreach ( $caption_positions as $pos_value => $pos_label ) : ?>
                <label>
                    <input 
                        type="radio" 
                        name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_caption_position]" 
                        value="<?php echo esc_attr( $pos_value ); ?>"
                        <?php checked( $value, $pos_value ); ?>
                    />
                    <?php echo esc_html( $pos_label ); ?>
                </label>
            <?php endforeach; ?>
        </div>
        <?php
    }

    /**
     * Rendre le champ police caption
     */
    public function render_field_default_caption_font() {
        $options       = get_option( self::OPTION_NAME, array() );
        $value         = isset( $options['default_caption_font'] ) ? $options['default_caption_font'] : 'poppins';
        $caption_fonts = Fliz_API_Client::get_enums()['caption_fonts'];
        ?>
        <select name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_caption_font]">
            <?php foreach ( $caption_fonts as $font_value => $font_label ) : ?>
                <option value="<?php echo esc_attr( $font_value ); ?>" <?php selected( $value, $font_value ); ?>>
                    <?php echo esc_html( $font_label ); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php
    }

    /**
     * Rendre le champ couleur caption
     */
    public function render_field_default_caption_color() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = isset( $options['default_caption_color'] ) ? $options['default_caption_color'] : '#FFFFFF';
        ?>
        <input 
            type="text" 
            name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_caption_color]" 
            value="<?php echo esc_attr( $value ); ?>" 
            class="fliz-color-picker"
            data-default-color="#FFFFFF"
        />
        <?php
    }

    /**
     * Rendre le champ majuscules caption
     */
    public function render_field_default_caption_uppercase() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = ! empty( $options['default_caption_uppercase'] );
        ?>
        <label>
            <input 
                type="checkbox" 
                name="<?php echo esc_attr( self::OPTION_NAME ); ?>[default_caption_uppercase]" 
                value="1"
                <?php checked( $value ); ?>
            />
            <?php esc_html_e( 'Force uppercase captions', 'fliz-video-generator' ); ?>
        </label>
        <?php
    }

    /**
     * Rendre le champ mode d'animation
     */
    public function render_field_video_animation_mode() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = isset( $options['video_animation_mode'] ) ? $options['video_animation_mode'] : '';
        ?>
        <select name="<?php echo esc_attr( self::OPTION_NAME ); ?>[video_animation_mode]">
            <option value="" <?php selected( $value, '' ); ?>><?php esc_html_e( 'Classic (slideshow)', 'fliz-video-generator' ); ?></option>
            <option value="hook_only" <?php selected( $value, 'hook_only' ); ?>><?php esc_html_e( 'Video hook only', 'fliz-video-generator' ); ?></option>
            <option value="full_video" <?php selected( $value, 'full_video' ); ?>><?php esc_html_e( 'Full video', 'fliz-video-generator' ); ?></option>
        </select>
        <p class="description">
            <?php esc_html_e( 'Choose how animations are applied to your video.', 'fliz-video-generator' ); ?>
        </p>
        <?php
    }

    /**
     * Rendre le champ watermark
     */
    public function render_field_watermark_option() {
        $options        = get_option( self::OPTION_NAME, array() );
        $watermark_type = isset( $options['watermark_type'] ) ? $options['watermark_type'] : 'fliz';
        $watermark_url  = isset( $options['watermark_url'] ) ? $options['watermark_url'] : '';
        ?>
        <div class="fliz-watermark-options">
            <label class="fliz-watermark-option">
                <input 
                    type="radio" 
                    name="<?php echo esc_attr( self::OPTION_NAME ); ?>[watermark_type]" 
                    value="fliz"
                    class="fliz-watermark-radio"
                    <?php checked( $watermark_type, 'fliz' ); ?>
                />
                <span class="fliz-watermark-label">
                    <?php esc_html_e( 'Fliz Watermark', 'fliz-video-generator' ); ?>
                    <span class="fliz-watermark-price"><?php esc_html_e( 'Included', 'fliz-video-generator' ); ?></span>
                </span>
            </label>
            <label class="fliz-watermark-option">
                <input 
                    type="radio" 
                    name="<?php echo esc_attr( self::OPTION_NAME ); ?>[watermark_type]" 
                    value="none"
                    class="fliz-watermark-radio"
                    <?php checked( $watermark_type, 'none' ); ?>
                />
                <span class="fliz-watermark-label">
                    <?php esc_html_e( 'No Watermark', 'fliz-video-generator' ); ?>
                    <span class="fliz-watermark-price">+5 <?php esc_html_e( 'credits', 'fliz-video-generator' ); ?></span>
                </span>
            </label>
            <label class="fliz-watermark-option">
                <input 
                    type="radio" 
                    name="<?php echo esc_attr( self::OPTION_NAME ); ?>[watermark_type]" 
                    value="custom"
                    class="fliz-watermark-radio"
                    <?php checked( $watermark_type, 'custom' ); ?>
                />
                <span class="fliz-watermark-label">
                    <?php esc_html_e( 'Custom Watermark', 'fliz-video-generator' ); ?>
                    <span class="fliz-watermark-price">+10 <?php esc_html_e( 'credits', 'fliz-video-generator' ); ?></span>
                </span>
            </label>
        </div>
        
        <div class="fliz-custom-watermark-upload" id="fliz-custom-watermark-upload" style="<?php echo esc_attr( 'custom' !== $watermark_type ? 'display:none;' : '' ); ?>">
            <div class="fliz-watermark-preview-wrapper">
                <?php if ( ! empty( $watermark_url ) ) : ?>
                    <img src="<?php echo esc_url( $watermark_url ); ?>" alt="Watermark" class="fliz-watermark-preview" id="fliz-watermark-preview" />
                <?php else : ?>
                    <div class="fliz-watermark-placeholder" id="fliz-watermark-placeholder">
                        <span class="dashicons dashicons-format-image"></span>
                        <span><?php esc_html_e( 'No image selected', 'fliz-video-generator' ); ?></span>
                    </div>
                <?php endif; ?>
            </div>
            <input 
                type="hidden" 
                name="<?php echo esc_attr( self::OPTION_NAME ); ?>[watermark_url]" 
                value="<?php echo esc_url( $watermark_url ); ?>" 
                id="fliz-watermark-url"
            />
            <button type="button" class="button" id="fliz-upload-watermark">
                <span class="dashicons dashicons-upload"></span>
                <?php esc_html_e( 'Upload Image', 'fliz-video-generator' ); ?>
            </button>
            <?php if ( ! empty( $watermark_url ) ) : ?>
                <button type="button" class="button fliz-remove-watermark" id="fliz-remove-watermark">
                    <span class="dashicons dashicons-trash"></span>
                </button>
            <?php endif; ?>
            <p class="description">
                <?php esc_html_e( 'Recommended: PNG with transparent background, minimum 200x200px.', 'fliz-video-generator' ); ?>
            </p>
        </div>
        <?php
    }

    /**
     * Rendre le champ auto-generate
     */
    public function render_field_auto_generate() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = ! empty( $options['auto_generate'] );
        ?>
        <label>
            <input 
                type="checkbox" 
                name="<?php echo esc_attr( self::OPTION_NAME ); ?>[auto_generate]" 
                value="1"
                <?php checked( $value ); ?>
            />
            <?php esc_html_e( 'Automatically generate a video when a post is published', 'fliz-video-generator' ); ?>
        </label>
        <p class="description">
            <?php esc_html_e( 'When enabled, a video will be generated automatically for each newly published post.', 'fliz-video-generator' ); ?>
        </p>
        <?php
    }

    /**
     * Rendre le champ auto-insert shortcode
     */
    public function render_field_auto_insert_shortcode() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = isset( $options['auto_insert_shortcode'] ) ? $options['auto_insert_shortcode'] : true;
        ?>
        <label>
            <input 
                type="checkbox" 
                name="<?php echo esc_attr( self::OPTION_NAME ); ?>[auto_insert_shortcode]" 
                value="1"
                <?php checked( $value ); ?>
            />
            <?php esc_html_e( 'Automatically add the video shortcode at the end of the article', 'fliz-video-generator' ); ?>
        </label>
        <p class="description">
            <?php esc_html_e( 'When enabled, the [fliz_video] shortcode will be automatically inserted at the end of the post content when the video is ready.', 'fliz-video-generator' ); ?>
        </p>
        <?php
    }

    /**
     * Rendre le champ is_automatic
     */
    public function render_field_is_automatic() {
        $options = get_option( self::OPTION_NAME, array() );
        $value   = isset( $options['is_automatic'] ) ? $options['is_automatic'] : true;
        ?>
        <label>
            <input 
                type="checkbox" 
                name="<?php echo esc_attr( self::OPTION_NAME ); ?>[is_automatic]" 
                value="1"
                <?php checked( $value ); ?>
            />
            <?php esc_html_e( 'Fully automatic video generation (recommended)', 'fliz-video-generator' ); ?>
        </label>
        <p class="description">
            <?php esc_html_e( 'Let Fliz AI handle the entire video creation process automatically.', 'fliz-video-generator' ); ?>
        </p>
        <?php
    }

    /**
     * Test API via AJAX
     */
    public function ajax_test_api() {
        check_ajax_referer( 'fliz_settings_nonce', 'security' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'fliz-video-generator' ) ) );
        }

        $api_key = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';

        $client = new Fliz_API_Client( $api_key );
        $result = $client->test_connection();

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( array( 'message' => $result->get_error_message() ) );
        }

        wp_send_json_success(
            array(
                'message' => __( 'Connection successful!', 'fliz-video-generator' ),
            )
        );
    }

    /**
     * Rafraîchir les options API via AJAX
     */
    public function ajax_refresh_options() {
        check_ajax_referer( 'fliz_settings_nonce', 'security' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'fliz-video-generator' ) ) );
        }

        $client = new Fliz_API_Client();

        $voices = $client->get_voices();
        $musics = $client->get_musics();

        if ( is_wp_error( $voices ) ) {
            wp_send_json_error( array( 'message' => $voices->get_error_message() ) );
        }

        wp_send_json_success(
            array(
                'voices' => $voices,
                'musics' => is_wp_error( $musics ) ? array() : $musics,
            )
        );
    }
}
