<?php
/**
 * Classe Fliz_Metabox
 *
 * Gère la metabox dans l'éditeur d'articles.
 *
 * @package Fliz_Video_Generator
 */

// Empêcher l'accès direct
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Classe Fliz_Metabox
 */
class Fliz_Metabox {

    /**
     * Initialiser
     */
    public function init() {
        add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
        add_action( 'save_post', array( $this, 'save_meta_box' ), 10, 2 );
        add_action( 'wp_ajax_fliz_generate_video', array( $this, 'ajax_generate_video' ) );
        add_action( 'wp_ajax_fliz_check_video_status', array( $this, 'ajax_check_status' ) );
        add_action( 'wp_ajax_fliz_regenerate_video', array( $this, 'ajax_regenerate_video' ) );
    }

    /**
     * Ajouter la metabox
     */
    public function add_meta_box() {
        $post_types = apply_filters( 'fliz_supported_post_types', array( 'post' ) );

        foreach ( $post_types as $post_type ) {
            add_meta_box(
                'fliz-video-generator',
                __( 'Fliz Video Generator', 'fliz-video-generator' ),
                array( $this, 'render_meta_box' ),
                $post_type,
                'side',
                'high'
            );
        }
    }

    /**
     * Afficher la metabox
     *
     * @param WP_Post $post Article courant.
     */
    public function render_meta_box( $post ) {
        // Vérifier si l'API est configurée
        $options = get_option( 'fliz_settings', array() );
        if ( empty( $options['api_key'] ) ) {
            $this->render_no_api_key();
            return;
        }

        // Récupérer les données existantes
        $video_id  = get_post_meta( $post->ID, '_fliz_video_id', true );
        $video_url = get_post_meta( $post->ID, '_fliz_video_url', true );
        $status    = get_post_meta( $post->ID, '_fliz_video_status', true );

        // Nonce
        wp_nonce_field( 'fliz_metabox_nonce', 'fliz_metabox_nonce_field' );

        // Inclure le template
        include FLIZ_PLUGIN_PATH . 'includes/admin/partials/metabox-display.php';
    }

    /**
     * Afficher le message pas de clé API
     */
    private function render_no_api_key() {
        ?>
        <div class="fliz-no-api-key">
            <div class="fliz-no-api-key-icon">
                <?php Fliz_Settings::echo_menu_icon_svg(); ?>
            </div>
            <h4><?php esc_html_e( 'Configuration Required', 'fliz-video-generator' ); ?></h4>
            <p><?php esc_html_e( 'To generate videos, please configure your Fliz API key first.', 'fliz-video-generator' ); ?></p>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=fliz-settings' ) ); ?>" class="button button-primary">
                <span class="dashicons dashicons-admin-settings"></span>
                <?php esc_html_e( 'Configure Fliz', 'fliz-video-generator' ); ?>
            </a>
        </div>
        <?php
    }

    /**
     * Sauvegarder les données de la metabox
     *
     * @param int     $post_id ID de l'article.
     * @param WP_Post $post    Objet article.
     */
    public function save_meta_box( $post_id, $post ) {
        // Vérifier le nonce
        if ( ! isset( $_POST['fliz_metabox_nonce_field'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['fliz_metabox_nonce_field'] ) ), 'fliz_metabox_nonce' ) ) {
            return;
        }

        // Vérifier les permissions
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        // Ne pas sauvegarder pour les révisions ou autosaves
        if ( wp_is_post_autosave( $post_id ) || wp_is_post_revision( $post_id ) ) {
            return;
        }

        // Sauvegarder l'option auto-generate
        if ( isset( $_POST['fliz_auto_generate'] ) ) {
            update_post_meta( $post_id, '_fliz_auto_generate', true );
        } else {
            delete_post_meta( $post_id, '_fliz_auto_generate' );
        }
    }

    /**
     * Générer une vidéo via AJAX
     */
    public function ajax_generate_video() {
        check_ajax_referer( 'fliz_metabox_ajax', 'security' );

        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'fliz-video-generator' ) ) );
        }

        // Vérifier la clé API
        $options = get_option( 'fliz_settings', array() );
        if ( empty( $options['api_key'] ) ) {
            wp_send_json_error(
                array(
                    'message'      => __( 'API key not configured. Please configure your Fliz API key in the plugin settings.', 'fliz-video-generator' ),
                    'settings_url' => admin_url( 'admin.php?page=fliz-settings' ),
                    'need_config'  => true,
                )
            );
        }

        $post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;
        if ( ! $post_id ) {
            wp_send_json_error( array( 'message' => __( 'Invalid post ID.', 'fliz-video-generator' ) ) );
        }

        $post = get_post( $post_id );
        if ( ! $post ) {
            wp_send_json_error( array( 'message' => __( 'Post not found.', 'fliz-video-generator' ) ) );
        }

        // Paramètres depuis le formulaire (ou valeurs par défaut)
        $format   = isset( $_POST['format'] ) ? sanitize_text_field( wp_unslash( $_POST['format'] ) ) : ( isset( $options['default_format'] ) ? $options['default_format'] : 'size_16_9' );
        $category = isset( $_POST['category'] ) ? sanitize_text_field( wp_unslash( $_POST['category'] ) ) : ( isset( $options['default_category'] ) ? $options['default_category'] : 'article' );

        // Construire les paramètres API Fliz
        $params = array(
            'name'              => $post->post_title,
            'description'       => wp_strip_all_tags( $post->post_content ),
            'category'          => $category,
            'format'            => $format,
            'lang'              => isset( $options['default_lang'] ) ? $options['default_lang'] : 'en',
            'script_style'      => isset( $options['default_script_style'] ) ? $options['default_script_style'] : null,
            'image_style'       => isset( $options['default_image_style'] ) ? $options['default_image_style'] : null,
            'caption_style'     => isset( $options['default_caption_style'] ) ? $options['default_caption_style'] : 'animated_background',
            'caption_position'  => isset( $options['default_caption_position'] ) ? $options['default_caption_position'] : 'bottom',
            'caption_font'      => isset( $options['default_caption_font'] ) ? $options['default_caption_font'] : 'poppins',
            'caption_color'     => isset( $options['default_caption_color'] ) ? $options['default_caption_color'] : null,
            'caption_uppercase' => isset( $options['default_caption_uppercase'] ) ? $options['default_caption_uppercase'] : null,
            'voice_id'          => isset( $options['default_voice_id'] ) ? $options['default_voice_id'] : null,
            'is_male_voice'     => self::get_is_male_voice_value( $options ),
            'music_id'          => isset( $options['default_music_id'] ) ? $options['default_music_id'] : null,
            'music_volume'      => isset( $options['default_music_volume'] ) ? $options['default_music_volume'] : 15,
            'site_url'          => get_permalink( $post_id ),
            'site_name'         => get_bloginfo( 'name' ),
            'is_automatic'      => isset( $options['is_automatic'] ) ? $options['is_automatic'] : true,
        );

        // Filtrer les paramètres
        $params = apply_filters( 'fliz_video_params', $params, $post_id );

        // Gestion du mode d'animation
        // - '' (vide/null) : classique slideshow
        // - 'hook_only' : accroche vidéo seulement
        // - 'full_video' : entièrement en vidéo
        $animation_mode = isset( $options['video_animation_mode'] ) ? $options['video_animation_mode'] : '';
        if ( ! empty( $animation_mode ) ) {
            $params['video_animation_mode'] = $animation_mode;
        }

        // Gestion du watermark selon le type
        // - fliz (défaut) : ne pas passer le paramètre -> watermark Fliz
        // - none : passer watermark_url = null -> pas de watermark
        // - custom : passer watermark_url = URL de l'image
        $watermark_type = isset( $options['watermark_type'] ) ? $options['watermark_type'] : 'fliz';
        if ( 'none' === $watermark_type ) {
            // Pas de watermark
            $params['watermark_url'] = null;
        } elseif ( 'custom' === $watermark_type && ! empty( $options['watermark_url'] ) ) {
            // Watermark personnalisé
            $params['watermark_url'] = $options['watermark_url'];
        }
        // Si 'fliz', on ne met pas le paramètre - l'API utilisera le watermark Fliz par défaut

        // Pour product/ad, collecter les images
        if ( in_array( $category, array( 'product', 'ad' ), true ) ) {
            $images = $this->collect_post_images( $post_id );
            if ( count( $images ) >= 3 ) {
                $params['image_urls'] = array_slice( $images, 0, 10 ); // Max 10 images
            }
        }

        // Appeler l'API
        $client = new Fliz_API_Client();
        $result = $client->create_video( $params );

        if ( is_wp_error( $result ) ) {
            // Logger l'erreur
            Fliz_Logger::log( $post_id, '', 'error', $result->get_error_message() );
            
            // Vérifier si c'est une erreur d'authentification
            $error_data = $result->get_error_data();
            $response   = array( 'message' => $result->get_error_message() );
            
            if ( is_array( $error_data ) && ! empty( $error_data['need_config'] ) ) {
                $response['need_config']  = true;
                $response['settings_url'] = $error_data['settings_url'];
            }
            
            wp_send_json_error( $response );
        }

        // Sauvegarder les métadonnées
        $video_id = isset( $result['video_id'] ) ? $result['video_id'] : '';
        update_post_meta( $post_id, '_fliz_video_id', $video_id );
        update_post_meta( $post_id, '_fliz_video_status', 'pending' );
        update_post_meta( $post_id, '_fliz_video_format', $format );
        update_post_meta( $post_id, '_fliz_video_category', $category );

        // Logger
        Fliz_Logger::log( $post_id, $video_id, 'pending', __( 'Video generation started.', 'fliz-video-generator' ) );

        wp_send_json_success(
            array(
                'video_id' => $video_id,
                'cost'     => isset( $result['cost'] ) ? $result['cost'] : 1,
                'status'   => 'pending',
                'message'  => __( 'Video generation started!', 'fliz-video-generator' ),
            )
        );
    }

    /**
     * Régénérer une vidéo via AJAX
     */
    public function ajax_regenerate_video() {
        check_ajax_referer( 'fliz_metabox_ajax', 'security' );

        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'fliz-video-generator' ) ) );
        }

        $post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;
        if ( ! $post_id ) {
            wp_send_json_error( array( 'message' => __( 'Invalid post ID.', 'fliz-video-generator' ) ) );
        }

        // Supprimer l'ancienne vidéo
        delete_post_meta( $post_id, '_fliz_video_id' );
        delete_post_meta( $post_id, '_fliz_video_url' );
        delete_post_meta( $post_id, '_fliz_video_status' );

        // Appeler la génération
        $this->ajax_generate_video();
    }

    /**
     * Collecter les images d'un article
     *
     * @param int $post_id ID de l'article.
     * @return array URLs des images.
     */
    private function collect_post_images( $post_id ) {
        $images = array();

        // Image mise en avant
        $thumbnail_id = get_post_thumbnail_id( $post_id );
        if ( $thumbnail_id ) {
            $images[] = wp_get_attachment_url( $thumbnail_id );
        }

        // Images dans le contenu
        $post = get_post( $post_id );
        preg_match_all( '/<img[^>]+src=["\']([^"\']+)["\']/', $post->post_content, $matches );
        if ( ! empty( $matches[1] ) ) {
            $images = array_merge( $images, $matches[1] );
        }

        // Galerie WooCommerce si disponible
        $gallery_ids = get_post_meta( $post_id, '_product_image_gallery', true );
        if ( $gallery_ids ) {
            foreach ( explode( ',', $gallery_ids ) as $id ) {
                $url = wp_get_attachment_url( $id );
                if ( $url ) {
                    $images[] = $url;
                }
            }
        }

        return array_unique( array_filter( $images ) );
    }

    /**
     * Vérifier le statut via AJAX
     */
    public function ajax_check_status() {
        check_ajax_referer( 'fliz_metabox_ajax', 'security' );

        $video_id = isset( $_POST['video_id'] ) ? sanitize_text_field( wp_unslash( $_POST['video_id'] ) ) : '';
        $post_id  = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

        if ( empty( $video_id ) ) {
            wp_send_json_error( array( 'message' => __( 'Invalid video ID.', 'fliz-video-generator' ) ) );
        }

        $client = new Fliz_API_Client();
        $result = $client->get_video_status( $video_id );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( array( 'message' => $result->get_error_message() ) );
        }

        $status = isset( $result['status'] ) ? $result['status'] : 'unknown';

        // Si terminé, sauvegarder l'URL
        if ( 'complete' === $status && ! empty( $result['url'] ) ) {
            update_post_meta( $post_id, '_fliz_video_url', esc_url_raw( $result['url'] ) );
            update_post_meta( $post_id, '_fliz_video_status', 'complete' );

            // Ajouter le shortcode à la fin de l'article
            $this->maybe_add_shortcode_to_content( $post_id, $video_id );

            // Logger
            Fliz_Logger::update_status( $video_id, 'complete', __( 'Video ready.', 'fliz-video-generator' ) );
        } elseif ( in_array( $status, array( 'failed', 'failed_unrecoverable', 'block' ), true ) ) {
            update_post_meta( $post_id, '_fliz_video_status', 'error' );
            $error_message = isset( $result['error'] ) ? $result['error'] : __( 'Generation failed.', 'fliz-video-generator' );
            Fliz_Logger::update_status( $video_id, 'error', $error_message );
        }

        wp_send_json_success( $result );
    }

    /**
     * Ajouter le shortcode à la fin du contenu de l'article
     *
     * @param int    $post_id  ID de l'article.
     * @param string $video_id ID de la vidéo Fliz.
     */
    private function maybe_add_shortcode_to_content( $post_id, $video_id ) {
        // Vérifier les options
        $options = get_option( 'fliz_settings', array() );
        $auto_insert = isset( $options['auto_insert_shortcode'] ) ? $options['auto_insert_shortcode'] : true;

        // Si l'option est désactivée, ne pas ajouter
        if ( ! $auto_insert ) {
            return;
        }

        // Récupérer l'article
        $post = get_post( $post_id );
        if ( ! $post ) {
            return;
        }

        $shortcode = '[fliz_video id="' . esc_attr( $video_id ) . '"]';

        // Vérifier si le shortcode n'est pas déjà présent
        if ( strpos( $post->post_content, '[fliz_video' ) !== false ) {
            return;
        }

        // Ajouter le shortcode à la fin du contenu
        $new_content = $post->post_content . "\n\n" . $shortcode;

        // Mettre à jour l'article sans déclencher les hooks
        wp_update_post(
            array(
                'ID'           => $post_id,
                'post_content' => $new_content,
            )
        );
    }

    /**
     * Obtenir la valeur is_male_voice depuis les options
     *
     * WordPress stocke les booléens false comme '' (chaîne vide),
     * donc on doit se baser sur voice_gender pour déterminer la vraie valeur.
     *
     * @param array $options Les options du plugin.
     * @return bool|null true pour homme, false pour femme, null pour auto.
     */
    private static function get_is_male_voice_value( $options ) {
        $voice_gender = isset( $options['default_voice_gender'] ) ? $options['default_voice_gender'] : 'auto';

        if ( 'male' === $voice_gender ) {
            return true;
        } elseif ( 'female' === $voice_gender ) {
            return false;
        }

        return null; // Auto
    }
}
