<?php
/**
 * Classe Fliz_Logs
 *
 * Gère la page de logs.
 *
 * @package Fliz_Video_Generator
 */

// Empêcher l'accès direct
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Classe Fliz_Logs
 */
class Fliz_Logs {

	/**
     * Page slug
	 */
	private const PAGE_SLUG = 'fliz-logs';

	/**
     * Initialiser
	 */
	public function init() {
		add_action( 'admin_menu', array( $this, 'add_submenu_page' ) );
		add_action( 'wp_ajax_fliz_refresh_log_status', array( $this, 'ajax_refresh_status' ) );
		add_action( 'wp_ajax_fliz_delete_log', array( $this, 'ajax_delete_log' ) );
	}

	/**
     * Ajouter la page de sous-menu
	 */
	public function add_submenu_page() {
		add_submenu_page(
			'fliz-settings',
            __( 'Generation Logs', 'fliz-video-generator' ),
			__( 'Logs', 'fliz-video-generator' ),
			'manage_options',
			self::PAGE_SLUG,
			array( $this, 'render_page' )
		);
	}

	/**
     * Afficher la page
	 */
	public function render_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

        // Récupérer le filtre
        $status_filter = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : 'all'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $page          = isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $per_page      = 20;

        // Arguments de requête
		$args = array(
            'limit'   => $per_page,
            'offset'  => ( $page - 1 ) * $per_page,
            'orderby' => 'created_at',
            'order'   => 'DESC',
		);

        if ( 'all' !== $status_filter ) {
            $args['status'] = $status_filter;
		}

		$logs   = Fliz_Logger::get_logs( $args );
		$counts = Fliz_Logger::count_by_status();

		include FLIZ_PLUGIN_PATH . 'includes/admin/partials/logs-page.php';
	}

	/**
     * Rafraîchir le statut d'un log via AJAX
	 */
	public function ajax_refresh_status() {
		check_ajax_referer( 'fliz_logs_nonce', 'security' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'fliz-video-generator' ) ) );
		}

		$video_id = isset( $_POST['video_id'] ) ? sanitize_text_field( wp_unslash( $_POST['video_id'] ) ) : '';

		if ( empty( $video_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid video ID.', 'fliz-video-generator' ) ) );
		}

		$client = new Fliz_API_Client();
		$result = $client->get_video_status( $video_id );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

        $status = isset( $result['status'] ) ? $result['status'] : 'unknown';

        // Mettre à jour le log
        if ( 'complete' === $status ) {
            Fliz_Logger::update_status( $video_id, 'complete', __( 'Video ready.', 'fliz-video-generator' ) );

            // Mettre à jour le post meta
		$log = Fliz_Logger::get_by_video_id( $video_id );
            if ( $log && ! empty( $result['url'] ) ) {
			update_post_meta( $log->post_id, '_fliz_video_url', esc_url_raw( $result['url'] ) );
			update_post_meta( $log->post_id, '_fliz_video_status', 'complete' );
            }
        } elseif ( in_array( $status, array( 'failed', 'failed_unrecoverable', 'block' ), true ) ) {
            $error_message = isset( $result['error'] ) ? $result['error'] : __( 'Generation failed.', 'fliz-video-generator' );
            Fliz_Logger::update_status( $video_id, 'error', $error_message );

            $log = Fliz_Logger::get_by_video_id( $video_id );
            if ( $log ) {
			update_post_meta( $log->post_id, '_fliz_video_status', 'error' );
		}
        }

        wp_send_json_success( $result );
	}

	/**
     * Supprimer un log via AJAX
	 */
	public function ajax_delete_log() {
		check_ajax_referer( 'fliz_logs_nonce', 'security' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'fliz-video-generator' ) ) );
		}

		$log_id = isset( $_POST['log_id'] ) ? absint( $_POST['log_id'] ) : 0;

		if ( ! $log_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid log ID.', 'fliz-video-generator' ) ) );
		}

        $result = Fliz_Logger::delete( $log_id );

        if ( $result ) {
		wp_send_json_success( array( 'message' => __( 'Log deleted.', 'fliz-video-generator' ) ) );
        } else {
            wp_send_json_error( array( 'message' => __( 'Failed to delete log.', 'fliz-video-generator' ) ) );
        }
	}
}
