<?php
/**
 * Classe Fliz_Bulk_Generator
 *
 * Gère la génération en masse de vidéos.
 *
 * @package Fliz_Video_Generator
 */

// Empêcher l'accès direct
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Classe Fliz_Bulk_Generator
 */
class Fliz_Bulk_Generator {

	/**
     * Page slug
	 */
    private const PAGE_SLUG = 'fliz-bulk';

	/**
     * Initialiser
	 */
	public function init() {
		add_action( 'admin_menu', array( $this, 'add_submenu_page' ) );
		add_action( 'wp_ajax_fliz_bulk_generate', array( $this, 'ajax_bulk_generate' ) );
        add_action( 'wp_ajax_fliz_get_posts_list', array( $this, 'ajax_get_posts_list' ) );
	}

	/**
     * Ajouter la page de sous-menu
	 */
	public function add_submenu_page() {
		add_submenu_page(
			'fliz-settings',
			__( 'Bulk Generator', 'fliz-video-generator' ),
			__( 'Bulk Generator', 'fliz-video-generator' ),
			'edit_posts',
			self::PAGE_SLUG,
			array( $this, 'render_page' )
		);
	}

	/**
     * Afficher la page
	 */
	public function render_page() {
		if ( ! current_user_can( 'edit_posts' ) ) {
			return;
		}
        include FLIZ_PLUGIN_PATH . 'includes/admin/partials/bulk-generator-page.php';
    }

    /**
     * Récupérer la liste des posts via AJAX
     */
    public function ajax_get_posts_list() {
        check_ajax_referer( 'fliz_bulk_nonce', 'security' );

        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'fliz-video-generator' ) ) );
        }

        $page       = isset( $_POST['page'] ) ? absint( $_POST['page'] ) : 1;
		$per_page   = 20;
        $filter     = isset( $_POST['filter'] ) ? sanitize_text_field( wp_unslash( $_POST['filter'] ) ) : 'all';
        $category   = isset( $_POST['category'] ) ? absint( $_POST['category'] ) : 0;
        $search     = isset( $_POST['search'] ) ? sanitize_text_field( wp_unslash( $_POST['search'] ) ) : '';

		$args = array(
			'post_type'      => 'post',
			'post_status'    => 'publish',
			'posts_per_page' => $per_page,
            'paged'          => $page,
			'orderby'        => 'date',
			'order'          => 'DESC',
		);

        // Filtre par catégorie
        if ( $category > 0 ) {
            $args['cat'] = $category;
		}

        // Recherche
        if ( ! empty( $search ) ) {
			$args['s'] = $search;
		}

        // Filtre avec/sans vidéo
        // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Meta query is necessary for filtering posts by video status.
        if ( 'with_video' === $filter ) {
			$args['meta_query'] = array(
				array(
					'key'     => '_fliz_video_id',
					'compare' => 'EXISTS',
				),
			);
        } elseif ( 'without_video' === $filter ) {
			$args['meta_query'] = array(
				array(
					'key'     => '_fliz_video_id',
					'compare' => 'NOT EXISTS',
				),
			);
		}
        // phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		$query = new WP_Query( $args );
        $posts = array();

        foreach ( $query->posts as $post ) {
            $video_id = get_post_meta( $post->ID, '_fliz_video_id', true );
            $status   = get_post_meta( $post->ID, '_fliz_video_status', true );

            $posts[] = array(
                'id'         => $post->ID,
                'title'      => $post->post_title,
                'author'     => get_the_author_meta( 'display_name', $post->post_author ),
                'date'       => get_the_date( 'Y-m-d', $post ),
                'categories' => wp_get_post_categories( $post->ID, array( 'fields' => 'names' ) ),
                'has_video'  => ! empty( $video_id ),
                'status'     => $status,
                'edit_url'   => get_edit_post_link( $post->ID, 'raw' ),
            );
        }

        wp_send_json_success(
				array(
                'posts'       => $posts,
                'total'       => $query->found_posts,
                'total_pages' => $query->max_num_pages,
                'current'     => $page,
            )
        );
    }

    /**
     * Générer des vidéos en masse via AJAX
	 */
	public function ajax_bulk_generate() {
		check_ajax_referer( 'fliz_bulk_nonce', 'security' );

		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'fliz-video-generator' ) ) );
		}

		$post_ids = isset( $_POST['post_ids'] ) ? array_map( 'absint', (array) $_POST['post_ids'] ) : array();
		$format   = isset( $_POST['format'] ) ? sanitize_text_field( wp_unslash( $_POST['format'] ) ) : 'size_16_9';
		$category = isset( $_POST['category'] ) ? sanitize_text_field( wp_unslash( $_POST['category'] ) ) : 'article';

		if ( empty( $post_ids ) ) {
			wp_send_json_error( array( 'message' => __( 'No posts selected.', 'fliz-video-generator' ) ) );
		}

        $options = get_option( 'fliz_settings', array() );
        $client  = new Fliz_API_Client();
        $results = array();

		foreach ( $post_ids as $post_id ) {
			$post = get_post( $post_id );
			if ( ! $post ) {
                $results[ $post_id ] = array(
                    'success' => false,
                    'message' => __( 'Post not found.', 'fliz-video-generator' ),
                );
				continue;
			}

            // Vérifier si une vidéo existe déjà
			$existing_video = get_post_meta( $post_id, '_fliz_video_id', true );
			if ( ! empty( $existing_video ) ) {
                $results[ $post_id ] = array(
                    'success' => false,
                    'message' => __( 'Video already exists.', 'fliz-video-generator' ),
				);
				continue;
			}

            // Paramètres
			$params = array(
				'name'              => $post->post_title,
				'description'       => wp_strip_all_tags( $post->post_content ),
				'category'          => $category,
				'format'            => $format,
                'lang'              => isset( $options['default_lang'] ) ? $options['default_lang'] : 'en',
                'script_style'      => isset( $options['default_script_style'] ) ? $options['default_script_style'] : null,
                'image_style'       => isset( $options['default_image_style'] ) ? $options['default_image_style'] : null,
                'caption_style'     => isset( $options['default_caption_style'] ) ? $options['default_caption_style'] : 'animated_background',
                'caption_position'  => isset( $options['default_caption_position'] ) ? $options['default_caption_position'] : 'bottom',
                'caption_font'      => isset( $options['default_caption_font'] ) ? $options['default_caption_font'] : 'poppins',
				'site_url'          => get_permalink( $post_id ),
				'site_name'         => get_bloginfo( 'name' ),
                'is_automatic'      => isset( $options['is_automatic'] ) ? $options['is_automatic'] : true,
                'is_male_voice'     => self::get_is_male_voice_value( $options ),
            );

            // Gestion du mode d'animation
            $animation_mode = isset( $options['video_animation_mode'] ) ? $options['video_animation_mode'] : '';
            if ( ! empty( $animation_mode ) ) {
                $params['video_animation_mode'] = $animation_mode;
            }

            // Gestion du watermark
            // - fliz (défaut) : ne pas passer le paramètre -> watermark Fliz
            // - none : passer watermark_url = null -> pas de watermark
            // - custom : passer watermark_url = URL de l'image
            $watermark_type = isset( $options['watermark_type'] ) ? $options['watermark_type'] : 'fliz';
            if ( 'none' === $watermark_type ) {
                // Pas de watermark
                $params['watermark_url'] = null;
            } elseif ( 'custom' === $watermark_type && ! empty( $options['watermark_url'] ) ) {
				$params['watermark_url'] = $options['watermark_url'];
			}
            // Si 'fliz', on ne met pas le paramètre - l'API utilisera le watermark Fliz par défaut

            // Appeler l'API
			$result = $client->create_video( $params );

			if ( is_wp_error( $result ) ) {
                Fliz_Logger::log( $post_id, '', 'error', $result->get_error_message() );
                $results[ $post_id ] = array(
                    'success' => false,
                    'message' => $result->get_error_message(),
                );
				continue;
			}

            // Sauvegarder
            $video_id = isset( $result['video_id'] ) ? $result['video_id'] : '';
			update_post_meta( $post_id, '_fliz_video_id', $video_id );
			update_post_meta( $post_id, '_fliz_video_status', 'pending' );

            Fliz_Logger::log( $post_id, $video_id, 'pending', __( 'Bulk generation started.', 'fliz-video-generator' ) );

            $results[ $post_id ] = array(
                'success'  => true,
				'video_id' => $video_id,
			);
		}

        // Planifier la vérification
        if ( ! wp_next_scheduled( 'fliz_check_pending_videos' ) ) {
			wp_schedule_single_event( time() + 60, 'fliz_check_pending_videos' );
		}

		wp_send_json_success(
			array(
                'results' => $results,
				'message' => sprintf(
                    /* translators: %d: number of videos */
                    __( '%d videos queued for generation.', 'fliz-video-generator' ),
                    count( array_filter( $results, function( $r ) { return $r['success']; } ) )
                ),
            )
        );
    }

    /**
     * Compter les posts avec/sans vidéo
     *
     * @return array
     */
    public static function get_counts() {
        // Essayer de récupérer depuis le cache
        $cache_key = 'fliz_bulk_counts';
        $counts    = wp_cache_get( $cache_key, 'fliz' );

        if ( false !== $counts ) {
            return $counts;
        }

        global $wpdb;

        // Total posts
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $total = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'post' AND post_status = 'publish'"
        );

        // Avec vidéo
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $with_video = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p 
                INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id 
                WHERE p.post_type = 'post' AND p.post_status = 'publish' 
                AND pm.meta_key = %s AND pm.meta_value != ''",
                '_fliz_video_id'
            )
        );

        $counts = array(
            'total'         => (int) $total,
            'with_video'    => (int) $with_video,
            'without_video' => (int) $total - (int) $with_video,
        );

        // Mettre en cache pour 5 minutes
        wp_cache_set( $cache_key, $counts, 'fliz', 300 );

        return $counts;
    }

    /**
     * Obtenir la valeur is_male_voice depuis les options
     *
     * WordPress stocke les booléens false comme '' (chaîne vide),
     * donc on doit se baser sur voice_gender pour déterminer la vraie valeur.
     *
     * @param array $options Les options du plugin.
     * @return bool|null true pour homme, false pour femme, null pour auto.
     */
    private static function get_is_male_voice_value( $options ) {
        $voice_gender = isset( $options['default_voice_gender'] ) ? $options['default_voice_gender'] : 'auto';

        if ( 'male' === $voice_gender ) {
            return true;
        } elseif ( 'female' === $voice_gender ) {
            return false;
        }

        return null; // Auto
    }
}
