<?php
/**
 * Classe Fliz_Auto_Generator
 *
 * Gère la génération automatique à la publication.
 *
 * @package Fliz_Video_Generator
 */

// Empêcher l'accès direct
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Classe Fliz_Auto_Generator
 */
class Fliz_Auto_Generator {

    /**
     * Initialiser
     */
    public function init() {
        add_action( 'transition_post_status', array( $this, 'handle_publish' ), 10, 3 );
        add_action( 'fliz_check_pending_videos', array( $this, 'check_pending_videos' ) );
    }

    /**
     * Gérer la publication
     *
     * @param string  $new_status Nouveau statut.
     * @param string  $old_status Ancien statut.
     * @param WP_Post $post       Article.
     */
    public function handle_publish( $new_status, $old_status, $post ) {
        // Vérifications
        if ( 'publish' !== $new_status ) {
            return;
        }
        if ( 'publish' === $old_status ) {
            return;
        }

        $post_types = apply_filters( 'fliz_supported_post_types', array( 'post' ) );
        if ( ! in_array( $post->post_type, $post_types, true ) ) {
            return;
        }

        // Vérifier si auto-génération activée
        $options       = get_option( 'fliz_settings', array() );
        $auto_generate = ! empty( $options['auto_generate'] );
        $post_auto     = get_post_meta( $post->ID, '_fliz_auto_generate', true );

        if ( ! $auto_generate && ! $post_auto ) {
            return;
        }

        // Vérifier si l'API est configurée
        if ( empty( $options['api_key'] ) ) {
            return;
        }

        // Vérifier si pas déjà de vidéo
        $existing_video = get_post_meta( $post->ID, '_fliz_video_id', true );
        if ( ! empty( $existing_video ) ) {
            return;
        }

        // Générer la vidéo
        $this->generate_video_for_post( $post->ID );

        // Planifier la vérification
        if ( ! wp_next_scheduled( 'fliz_check_pending_videos' ) ) {
            wp_schedule_single_event( time() + 60, 'fliz_check_pending_videos' );
        }
    }

    /**
     * Générer une vidéo pour un article
     *
     * @param int $post_id ID de l'article.
     */
    private function generate_video_for_post( $post_id ) {
        $options = get_option( 'fliz_settings', array() );
        $post    = get_post( $post_id );

        if ( ! $post ) {
            return;
        }

        $params = array(
            'name'              => $post->post_title,
            'description'       => wp_strip_all_tags( $post->post_content ),
            'category'          => isset( $options['default_category'] ) ? $options['default_category'] : 'article',
            'format'            => isset( $options['default_format'] ) ? $options['default_format'] : 'size_16_9',
            'lang'              => isset( $options['default_lang'] ) ? $options['default_lang'] : 'en',
            'script_style'      => isset( $options['default_script_style'] ) ? $options['default_script_style'] : null,
            'image_style'       => isset( $options['default_image_style'] ) ? $options['default_image_style'] : null,
            'caption_style'     => isset( $options['default_caption_style'] ) ? $options['default_caption_style'] : 'animated_background',
            'caption_position'  => isset( $options['default_caption_position'] ) ? $options['default_caption_position'] : 'bottom',
            'caption_font'      => isset( $options['default_caption_font'] ) ? $options['default_caption_font'] : 'poppins',
            'caption_color'     => isset( $options['default_caption_color'] ) ? $options['default_caption_color'] : null,
            'voice_id'          => isset( $options['default_voice_id'] ) ? $options['default_voice_id'] : null,
            'is_male_voice'     => self::get_is_male_voice_value( $options ),
            'music_id'          => isset( $options['default_music_id'] ) ? $options['default_music_id'] : null,
            'music_volume'      => isset( $options['default_music_volume'] ) ? $options['default_music_volume'] : 15,
            'site_url'          => get_permalink( $post_id ),
            'site_name'         => get_bloginfo( 'name' ),
            'is_automatic'      => isset( $options['is_automatic'] ) ? $options['is_automatic'] : true,
        );

        // Filtrer les paramètres
        $params = apply_filters( 'fliz_video_params', $params, $post_id );

        // Gestion du mode d'animation
        $animation_mode = isset( $options['video_animation_mode'] ) ? $options['video_animation_mode'] : '';
        if ( ! empty( $animation_mode ) ) {
            $params['video_animation_mode'] = $animation_mode;
        }

        // Gestion du watermark selon le type
        // - fliz (défaut) : ne pas passer le paramètre -> watermark Fliz
        // - none : passer watermark_url = null -> pas de watermark
        // - custom : passer watermark_url = URL de l'image
        $watermark_type = isset( $options['watermark_type'] ) ? $options['watermark_type'] : 'fliz';
        if ( 'none' === $watermark_type ) {
            $params['watermark_url'] = null;
        } elseif ( 'custom' === $watermark_type && ! empty( $options['watermark_url'] ) ) {
            $params['watermark_url'] = $options['watermark_url'];
        }

        $client = new Fliz_API_Client();
        $result = $client->create_video( $params );

        if ( is_wp_error( $result ) ) {
            Fliz_Logger::log( $post_id, '', 'error', $result->get_error_message() );
            return;
        }

        $video_id = isset( $result['video_id'] ) ? $result['video_id'] : '';
        update_post_meta( $post_id, '_fliz_video_id', $video_id );
        update_post_meta( $post_id, '_fliz_video_status', 'pending' );

        Fliz_Logger::log( $post_id, $video_id, 'pending', __( 'Auto-generation started.', 'fliz-video-generator' ) );
    }

    /**
     * Vérifier les vidéos en attente (cron)
     */
    public function check_pending_videos() {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $pending_videos = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT pm1.post_id, pm1.meta_value as video_id 
                FROM {$wpdb->postmeta} pm1
                INNER JOIN {$wpdb->postmeta} pm2 ON pm1.post_id = pm2.post_id
                WHERE pm1.meta_key = %s 
                AND pm2.meta_key = %s 
                AND pm2.meta_value = %s",
                '_fliz_video_id',
                '_fliz_video_status',
                'pending'
            )
        );

        if ( empty( $pending_videos ) ) {
            return;
        }

        $client        = new Fliz_API_Client();
        $still_pending = false;

        foreach ( $pending_videos as $video ) {
            $result = $client->get_video_status( $video->video_id );

            if ( is_wp_error( $result ) ) {
                continue;
            }

            $status = isset( $result['status'] ) ? $result['status'] : 'unknown';

            if ( 'complete' === $status ) {
                update_post_meta( $video->post_id, '_fliz_video_url', esc_url_raw( $result['url'] ) );
                update_post_meta( $video->post_id, '_fliz_video_status', 'complete' );

                // Ajouter le shortcode
                $this->add_shortcode_to_post( $video->post_id, $video->video_id );

                Fliz_Logger::update_status( $video->video_id, 'complete', __( 'Video ready.', 'fliz-video-generator' ) );
            } elseif ( in_array( $status, array( 'failed', 'failed_unrecoverable', 'block' ), true ) ) {
                update_post_meta( $video->post_id, '_fliz_video_status', 'error' );
                $error_message = isset( $result['error'] ) ? $result['error'] : __( 'Generation failed.', 'fliz-video-generator' );
                Fliz_Logger::update_status( $video->video_id, 'error', $error_message );
            } else {
                $still_pending = true;
            }
        }

        // Replanifier si encore des vidéos en attente
        if ( $still_pending && ! wp_next_scheduled( 'fliz_check_pending_videos' ) ) {
            wp_schedule_single_event( time() + 60, 'fliz_check_pending_videos' );
        }
    }

    /**
     * Ajouter le shortcode au contenu du post
     *
     * @param int    $post_id  ID du post.
     * @param string $video_id ID de la vidéo.
     */
    private function add_shortcode_to_post( $post_id, $video_id ) {
        $post      = get_post( $post_id );
        $shortcode = '[fliz_video id="' . esc_attr( $video_id ) . '"]';

        // Vérifier si le shortcode n'est pas déjà présent
        if ( false === strpos( $post->post_content, '[fliz_video' ) ) {
            $updated_content = $post->post_content . "\n\n" . $shortcode;

            // Utiliser wp_update_post sans déclencher les hooks pour éviter les boucles
            remove_action( 'transition_post_status', array( $this, 'handle_publish' ), 10 );

            wp_update_post(
                array(
                    'ID'           => $post_id,
                    'post_content' => $updated_content,
                )
            );

            add_action( 'transition_post_status', array( $this, 'handle_publish' ), 10, 3 );
        }
    }

    /**
     * Obtenir la valeur is_male_voice depuis les options
     *
     * WordPress stocke les booléens false comme '' (chaîne vide),
     * donc on doit se baser sur voice_gender pour déterminer la vraie valeur.
     *
     * @param array $options Les options du plugin.
     * @return bool|null true pour homme, false pour femme, null pour auto.
     */
    private static function get_is_male_voice_value( $options ) {
        $voice_gender = isset( $options['default_voice_gender'] ) ? $options['default_voice_gender'] : 'auto';

        if ( 'male' === $voice_gender ) {
            return true;
        } elseif ( 'female' === $voice_gender ) {
            return false;
        }

        return null; // Auto
    }
}

