/**
 * Fliz Video Generator - Scripts Page Réglages
 *
 * @package Fliz_Video_Generator
 */

/* global jQuery, flizSettings, wp */

(function($) {
    'use strict';

    var FlizSettings = {
        /**
         * Initialisation
         */
        init: function() {
            this.bindEvents();
            this.initColorPicker();
            this.initRangeSlider();
            
            // Vérifier l'état initial du watermark
            this.checkWatermarkState();
        },

        /**
         * Vérifier l'état du watermark
         */
        checkWatermarkState: function() {
            var $checked = $('.fliz-watermark-radio:checked');
            var $uploadSection = $('#fliz-custom-watermark-upload');
            
            if ($checked.val() === 'custom') {
                $uploadSection.show();
            } else {
                $uploadSection.hide();
            }
        },

        /**
         * Attacher les événements
         */
        bindEvents: function() {
            // Test API
            $(document).on('click', '#fliz-test-api-btn', this.testApiConnection.bind(this));
            
            // Toggle password visibility
            $(document).on('click', '.fliz-toggle-password', this.togglePassword.bind(this));
            
            // Range slider
            $(document).on('input', '#fliz-music-volume', this.updateRangeValue.bind(this));
            
            // Watermark options
            $(document).on('change', '.fliz-watermark-radio', this.toggleWatermarkUpload.bind(this));
            $(document).on('click', '#fliz-upload-watermark', this.uploadWatermark.bind(this));
            $(document).on('click', '#fliz-remove-watermark', this.removeWatermark.bind(this));
        },

        /**
         * Initialiser le color picker
         */
        initColorPicker: function() {
            if ($.fn.wpColorPicker) {
                $('.fliz-color-picker').wpColorPicker();
            }
        },

        /**
         * Initialiser le range slider
         */
        initRangeSlider: function() {
            var $slider = $('#fliz-music-volume');
            var $value = $('#fliz-music-volume-value');
            
            if ($slider.length && $value.length) {
                $value.text($slider.val() + '%');
            }
        },

        /**
         * Mettre à jour la valeur du range
         */
        updateRangeValue: function(e) {
            var $slider = $(e.currentTarget);
            var $value = $('#fliz-music-volume-value');
            $value.text($slider.val() + '%');
        },

        /**
         * Toggle la visibilité du mot de passe
         */
        togglePassword: function(e) {
            e.preventDefault();
            
            var $btn = $(e.currentTarget);
            var $input = $btn.siblings('input');
            var $icon = $btn.find('.dashicons');
            
            if ($input.attr('type') === 'password') {
                $input.attr('type', 'text');
                $icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            } else {
                $input.attr('type', 'password');
                $icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            }
        },

        /**
         * Tester la connexion API
         */
        testApiConnection: function(e) {
            e.preventDefault();
            
            var $btn = $(e.currentTarget);
            var $status = $('#fliz-api-status');
            var apiKey = $('#fliz_api_key').val();
            
            if (!apiKey) {
                $status.removeClass('success').addClass('error')
                    .html('<span class="dashicons dashicons-warning"></span> ' + flizSettings.strings.error);
                return;
            }
            
            // État de chargement
            $btn.prop('disabled', true);
            $status.removeClass('success error')
                .html('<span class="fliz-spinner"></span> ' + flizSettings.strings.testing);
            
            $.ajax({
                url: flizSettings.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fliz_test_api',
                    security: flizSettings.nonce,
                    api_key: apiKey
                },
                success: function(response) {
                    if (response.success) {
                        $status.removeClass('error').addClass('success')
                            .html('<span class="dashicons dashicons-yes-alt"></span> ' + flizSettings.strings.success);
                    } else {
                        $status.removeClass('success').addClass('error')
                            .html('<span class="dashicons dashicons-warning"></span> ' + (response.data.message || flizSettings.strings.error));
                    }
                },
                error: function() {
                    $status.removeClass('success').addClass('error')
                        .html('<span class="dashicons dashicons-warning"></span> ' + flizSettings.strings.error);
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        },

        /**
         * Toggle l'affichage de l'upload watermark
         */
        toggleWatermarkUpload: function(e) {
            var $radio = $(e.currentTarget);
            var $uploadSection = $('#fliz-custom-watermark-upload');
            
            if ($radio.val() === 'custom') {
                $uploadSection.slideDown(200);
            } else {
                $uploadSection.slideUp(200);
            }
        },

        /**
         * Upload du watermark
         */
        uploadWatermark: function(e) {
            e.preventDefault();
            
            var self = this;
            
            // Vérifier que wp.media est disponible
            if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
                console.error('wp.media is not available');
                alert('Media uploader not available. Please refresh the page.');
                return;
            }
            
            // Si le media frame existe déjà, le rouvrir
            if (this.watermarkFrame) {
                this.watermarkFrame.open();
                return;
            }
            
            // Créer le media frame
            this.watermarkFrame = wp.media({
                title: flizSettings.strings.selectImage || 'Select Watermark Image',
                button: {
                    text: flizSettings.strings.useImage || 'Use this image'
                },
                multiple: false,
                library: {
                    type: 'image'
                }
            });
            
            // Quand une image est sélectionnée
            this.watermarkFrame.on('select', function() {
                var attachment = self.watermarkFrame.state().get('selection').first().toJSON();
                
                // Mettre à jour l'URL cachée
                $('#fliz-watermark-url').val(attachment.url);
                
                // Afficher la preview
                var $wrapper = $('.fliz-watermark-preview-wrapper');
                $wrapper.html('<img src="' + attachment.url + '" alt="Watermark" class="fliz-watermark-preview" id="fliz-watermark-preview" />');
                
                // Afficher le bouton supprimer s'il n'existe pas
                if ($('#fliz-remove-watermark').length === 0) {
                    $('#fliz-upload-watermark').after(
                        '<button type="button" class="button fliz-remove-watermark" id="fliz-remove-watermark">' +
                        '<span class="dashicons dashicons-trash"></span>' +
                        '</button>'
                    );
                }
            });
            
            this.watermarkFrame.open();
        },

        /**
         * Supprimer le watermark
         */
        removeWatermark: function(e) {
            e.preventDefault();
            
            // Vider l'URL
            $('#fliz-watermark-url').val('');
            
            // Remettre le placeholder
            var $wrapper = $('.fliz-watermark-preview-wrapper');
            $wrapper.html(
                '<div class="fliz-watermark-placeholder" id="fliz-watermark-placeholder">' +
                '<span class="dashicons dashicons-format-image"></span>' +
                '<span>' + (flizSettings.strings.noImage || 'No image selected') + '</span>' +
                '</div>'
            );
            
            // Supprimer le bouton
            $('#fliz-remove-watermark').remove();
        }
    };

    // Initialiser quand le DOM est prêt
    $(document).ready(function() {
        FlizSettings.init();
    });

})(jQuery);
