/**
 * Fliz Video Generator - Scripts Metabox
 *
 * @package Fliz_Video_Generator
 */

/* global jQuery, flizMetabox, FlizAdmin */

(function($) {
    'use strict';

    var FlizMetabox = {
        pollingInterval: null,
        pollingDelay: 5000, // 5 secondes
        lastFormat: null,
        lastCategory: null,

        /**
         * Initialisation
         */
        init: function() {
            this.bindEvents();
            this.checkPendingVideo();
            this.storeInitialValues();
        },

        /**
         * Stocker les valeurs initiales du formulaire
         */
        storeInitialValues: function() {
            var $container = $('.fliz-metabox-content');
            var $format = $container.find('input[name="fliz_format"]:checked');
            
            if ($format.length) {
                this.lastFormat = $format.val();
            }
            // Catégorie toujours "article"
            this.lastCategory = 'article';
        },

        /**
         * Attacher les événements
         */
        bindEvents: function() {
            $(document).on('click', '#fliz-generate-btn', this.generateVideo.bind(this));
            $(document).on('click', '.fliz-copy-shortcode', this.copyShortcode.bind(this));
            $(document).on('click', '#fliz-regenerate-btn', this.regenerateVideo.bind(this));
            $(document).on('click', '#fliz-retry-btn', this.retryGeneration.bind(this));
        },

        /**
         * Vérifier si une vidéo est en attente
         */
        checkPendingVideo: function() {
            var $container = $('.fliz-metabox-content');
            var $loading = $container.find('.fliz-loading-state');
            
            if ($loading.length) {
                var videoId = $loading.data('video-id');
                if (videoId) {
                    this.startPolling(videoId);
                }
            }
        },

        /**
         * Générer une vidéo
         */
        generateVideo: function(e) {
            e.preventDefault();

            var $button = $(e.currentTarget);
            var $container = $button.closest('.fliz-metabox-content');

            // Vérifier que l'article est sauvegardé
            if (!flizMetabox.postId || flizMetabox.postId === 0) {
                this.showError($container, flizMetabox.strings.saveFirst);
                return;
            }

            var format = $container.find('input[name="fliz_format"]:checked').val();
            var category = 'article'; // Toujours "article"

            // Stocker les valeurs pour le retry
            this.lastFormat = format;
            this.lastCategory = category;

            // Désactiver le bouton
            $button.prop('disabled', true);
            this.showLoading($container);

            // Appel AJAX avec timeout étendu
            $.ajax({
                url: flizMetabox.ajaxUrl,
                type: 'POST',
                timeout: 60000, // 60 secondes timeout
                data: {
                    action: 'fliz_generate_video',
                    security: flizMetabox.nonce,
                    post_id: flizMetabox.postId,
                    format: format,
                    category: category
                },
                success: function(response) {
                    if (response.success) {
                        this.startPolling(response.data.video_id);
                    } else {
                        var settingsUrl = response.data.need_config ? response.data.settings_url : null;
                        this.showError($container, response.data.message || flizMetabox.strings.error, settingsUrl);
                    }
                }.bind(this),
                error: function(xhr, status, error) {
                    var message = flizMetabox.strings.error;
                    var settingsUrl = null;
                    
                    if (status === 'timeout') {
                        message = flizMetabox.strings.timeout || 'Request timed out. Please try again.';
                    } else if (xhr.responseJSON && xhr.responseJSON.data) {
                        message = xhr.responseJSON.data.message || message;
                        if (xhr.responseJSON.data.need_config) {
                            settingsUrl = xhr.responseJSON.data.settings_url;
                        }
                    }
                    this.showError($container, message, settingsUrl);
                    console.error('Fliz AJAX Error:', status, error, xhr.responseText);
                }.bind(this)
            });
        },

        /**
         * Réessayer la génération (depuis le bouton Retry)
         */
        retryGeneration: function(e) {
            e.preventDefault();

            var $container = $('.fliz-metabox-content');

            // Vérifier que l'article est sauvegardé
            if (!flizMetabox.postId || flizMetabox.postId === 0) {
                this.showError($container, flizMetabox.strings.saveFirst);
                return;
            }
            
            // Afficher le loader
            this.showLoading($container);

            // Appel AJAX avec les dernières valeurs stockées
            $.ajax({
                url: flizMetabox.ajaxUrl,
                type: 'POST',
                timeout: 60000, // 60 secondes timeout
                data: {
                    action: 'fliz_generate_video',
                    security: flizMetabox.nonce,
                    post_id: flizMetabox.postId,
                    format: this.lastFormat,
                    category: this.lastCategory
                },
                success: function(response) {
                    if (response.success) {
                        this.startPolling(response.data.video_id);
                    } else {
                        var settingsUrl = response.data.need_config ? response.data.settings_url : null;
                        this.showError($container, response.data.message || flizMetabox.strings.error, settingsUrl);
                    }
                }.bind(this),
                error: function(xhr, status, error) {
                    var message = flizMetabox.strings.error;
                    var settingsUrl = null;
                    
                    if (status === 'timeout') {
                        message = flizMetabox.strings.timeout || 'Request timed out. Please try again.';
                    } else if (xhr.responseJSON && xhr.responseJSON.data) {
                        message = xhr.responseJSON.data.message || message;
                        if (xhr.responseJSON.data.need_config) {
                            settingsUrl = xhr.responseJSON.data.settings_url;
                        }
                    }
                    this.showError($container, message, settingsUrl);
                    console.error('Fliz AJAX Error:', status, error, xhr.responseText);
                }.bind(this)
            });
        },

        /**
         * Régénérer une vidéo (depuis une vidéo existante)
         */
        regenerateVideo: function(e) {
            e.preventDefault();

            var $container = $('.fliz-metabox-content');
            
            // Afficher le loader
            this.showLoading($container);

            // Appel AJAX
            $.ajax({
                url: flizMetabox.ajaxUrl,
                type: 'POST',
                timeout: 60000, // 60 secondes timeout
                data: {
                    action: 'fliz_regenerate_video',
                    security: flizMetabox.nonce,
                    post_id: flizMetabox.postId
                },
                success: function(response) {
                    if (response.success) {
                        this.startPolling(response.data.video_id);
                    } else {
                        this.showError($container, response.data.message || flizMetabox.strings.error);
                    }
                }.bind(this),
                error: function(xhr, status, error) {
                    var message = flizMetabox.strings.error;
                    if (status === 'timeout') {
                        message = flizMetabox.strings.timeout || 'Request timed out. Please try again.';
                    }
                    this.showError($container, message);
                    console.error('Fliz AJAX Error:', status, error, xhr.responseText);
                }.bind(this)
            });
        },

        /**
         * Démarrer le polling du statut
         */
        startPolling: function(videoId) {
            var self = this;
            
            // Arrêter tout polling existant
            if (this.pollingInterval) {
                clearInterval(this.pollingInterval);
            }

            this.pollingInterval = setInterval(function() {
                $.ajax({
                    url: flizMetabox.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'fliz_check_video_status',
                        security: flizMetabox.nonce,
                        video_id: videoId,
                        post_id: flizMetabox.postId
                    },
                    success: function(response) {
                        if (response.success) {
                            var status = response.data.status;
                            
                            if (status === 'complete') {
                                self.stopPolling();
                                // Recharger la page pour afficher la vidéo
                                window.location.reload();
                            } else if (['failed', 'failed_unrecoverable', 'block'].indexOf(status) !== -1) {
                                self.stopPolling();
                                self.showError(
                                    $('.fliz-metabox-content'),
                                    response.data.error || flizMetabox.strings.error
                                );
                            }
                            // Sinon, continuer le polling
                        }
                    }
                });
            }, this.pollingDelay);
        },

        /**
         * Arrêter le polling
         */
        stopPolling: function() {
            if (this.pollingInterval) {
                clearInterval(this.pollingInterval);
                this.pollingInterval = null;
            }
        },

        /**
         * Afficher l'état de chargement
         */
        showLoading: function($container) {
            var html = '<div class="fliz-loading-state">' +
                '<div class="fliz-spinner"></div>' +
                '<p>' + flizMetabox.strings.generating + '</p>' +
                '<p class="fliz-status-text">' + flizMetabox.strings.checking + '</p>' +
                '</div>';
            $container.html(html);
        },

        /**
         * Afficher une erreur
         */
        showError: function($container, message, settingsUrl) {
            var html = '<div class="fliz-error-state">' +
                '<span class="dashicons dashicons-warning"></span>' +
                '<p>' + message + '</p>';
            
            if (settingsUrl) {
                // Erreur de configuration - afficher lien vers settings
                html += '<a href="' + settingsUrl + '" class="button button-primary">' +
                    '<span class="dashicons dashicons-admin-settings"></span> ' +
                    (flizMetabox.strings.configure || 'Configure Fliz') +
                    '</a>';
            } else {
                // Erreur normale - afficher bouton retry
                html += '<button type="button" class="button button-primary" id="fliz-retry-btn">' +
                    flizMetabox.strings.retry +
                    '</button>';
            }
            
            html += '</div>';
            $container.html(html);
        },

        /**
         * Copier le shortcode
         */
        copyShortcode: function(e) {
            e.preventDefault();
            
            var $btn = $(e.currentTarget);
            var shortcode = $btn.siblings('code').text();
            
            if (typeof FlizAdmin !== 'undefined' && FlizAdmin.copyToClipboard) {
                FlizAdmin.copyToClipboard(shortcode).then(function() {
                    this.showCopySuccess($btn);
                }.bind(this));
            } else if (navigator.clipboard) {
                navigator.clipboard.writeText(shortcode).then(function() {
                    this.showCopySuccess($btn);
                }.bind(this));
            } else {
                // Fallback
                var $temp = $('<input>');
                $('body').append($temp);
                $temp.val(shortcode).select();
                document.execCommand('copy');
                $temp.remove();
                this.showCopySuccess($btn);
            }
        },

        /**
         * Afficher le succès de copie
         */
        showCopySuccess: function($button) {
            var $icon = $button.find('.dashicons');
            $icon.removeClass('dashicons-clipboard').addClass('dashicons-yes');
            
            setTimeout(function() {
                $icon.removeClass('dashicons-yes').addClass('dashicons-clipboard');
            }, 2000);
        }
    };

    // Initialiser quand le DOM est prêt
    $(document).ready(function() {
        FlizMetabox.init();
    });

})(jQuery);
