/**
 * Fliz Video Generator - Scripts Bulk Generator
 *
 * @package Fliz_Video_Generator
 */

/* global jQuery, flizBulk */

(function($) {
    'use strict';

    var FlizBulk = {
        selectedPosts: [],
        currentPage: 1,
        currentFilter: 'all',
        currentCategory: 0,
        currentSearch: '',

        /**
         * Initialisation
         */
        init: function() {
            this.bindEvents();
            this.loadPosts();
        },

        /**
         * Attacher les événements
         */
        bindEvents: function() {
            // Filtres de statut
            $(document).on('click', '.fliz-filter-link', this.handleFilterClick.bind(this));
            
            // Filtre par catégorie
            $(document).on('click', '#fliz-apply-filters', this.applyFilters.bind(this));
            
            // Recherche
            var searchTimeout;
            $(document).on('input', '#fliz-search-posts', function() {
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(function() {
                    this.currentSearch = $('#fliz-search-posts').val();
                    this.currentPage = 1;
                    this.loadPosts();
                }.bind(this), 300);
            }.bind(this));
            
            // Sélection des posts
            $(document).on('change', '#fliz-select-all, #fliz-select-all-bottom', this.toggleSelectAll.bind(this));
            $(document).on('change', '.fliz-post-checkbox', this.handlePostSelect.bind(this));
            
            // Bouton de génération
            $(document).on('click', '#fliz-bulk-generate-btn', this.openModal.bind(this));
            
            // Modal
            $(document).on('click', '.fliz-modal-close, .fliz-modal-cancel', this.closeModal.bind(this));
            $(document).on('click', '#fliz-confirm-generate', this.startBulkGeneration.bind(this));
            $(document).on('click', '.fliz-modal', function(e) {
                if ($(e.target).hasClass('fliz-modal')) {
                    this.closeModal();
                }
            }.bind(this));
            
            // Pagination
            $(document).on('click', '#fliz-pagination .pagination-links a', this.handlePaginationClick.bind(this));
        },

        /**
         * Charger les posts
         */
        loadPosts: function() {
            var $tbody = $('#fliz-posts-list');
            
            $tbody.html(
                '<tr class="fliz-loading-row">' +
                '<td colspan="6">' +
                '<div class="fliz-spinner"></div>' +
                '<span>Loading posts...</span>' +
                '</td>' +
                '</tr>'
            );

            $.ajax({
                url: flizBulk.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fliz_get_posts_list',
                    security: flizBulk.nonce,
                    page: this.currentPage,
                    filter: this.currentFilter,
                    category: this.currentCategory,
                    search: this.currentSearch
                },
                success: function(response) {
                    if (response.success) {
                        this.renderPosts(response.data.posts);
                        this.renderPagination(response.data);
                        this.updateSelectAllState();
                    } else {
                        $tbody.html('<tr><td colspan="6">Error loading posts.</td></tr>');
                    }
                }.bind(this),
                error: function() {
                    $tbody.html('<tr><td colspan="6">Error loading posts.</td></tr>');
                }
            });
        },

        /**
         * Rendre les posts dans le tableau
         */
        renderPosts: function(posts) {
            var $tbody = $('#fliz-posts-list');
            
            if (!posts.length) {
                $tbody.html('<tr><td colspan="6" style="text-align:center;">No posts found.</td></tr>');
                return;
            }

            var html = '';
            
            posts.forEach(function(post) {
                var isSelected = this.selectedPosts.indexOf(post.id) !== -1;
                var videoIcon = post.has_video 
                    ? '<span class="fliz-video-status-icon has-video dashicons dashicons-yes-alt"></span>'
                    : '<span class="fliz-video-status-icon no-video dashicons dashicons-minus"></span>';
                
                html += '<tr class="fliz-post-row' + (isSelected ? ' selected' : '') + '" data-post-id="' + post.id + '">' +
                    '<td class="check-column">' +
                    '<input type="checkbox" class="fliz-post-checkbox" value="' + post.id + '"' + (isSelected ? ' checked' : '') + (post.has_video ? ' disabled title="Already has video"' : '') + '/>' +
                    '</td>' +
                    '<td class="column-title"><a href="' + post.edit_url + '">' + this.escapeHtml(post.title || '(no title)') + '</a></td>' +
                    '<td class="column-author">' + this.escapeHtml(post.author) + '</td>' +
                    '<td class="column-categories">' + this.escapeHtml(post.categories.join(', ')) + '</td>' +
                    '<td class="column-video">' + videoIcon + '</td>' +
                    '<td class="column-date">' + post.date + '</td>' +
                    '</tr>';
            }.bind(this));

            $tbody.html(html);
        },

        /**
         * Rendre la pagination
         */
        renderPagination: function(data) {
            var $pagination = $('#fliz-pagination');
            var $displayNum = $pagination.find('.displaying-num');
            var $links = $pagination.find('.pagination-links');

            $displayNum.text(data.total + ' items');

            if (data.total_pages <= 1) {
                $links.html('');
                return;
            }

            var html = '';
            var maxPages = Math.min(5, data.total_pages);
            var startPage = Math.max(1, data.current - 2);
            var endPage = Math.min(data.total_pages, startPage + maxPages - 1);

            // Première page
            if (data.current > 1) {
                html += '<a href="#" data-page="1">&laquo;</a>';
                html += '<a href="#" data-page="' + (data.current - 1) + '">&lsaquo;</a>';
            }

            // Pages
            for (var i = startPage; i <= endPage; i++) {
                if (i === data.current) {
                    html += '<span class="current">' + i + '</span>';
                } else {
                    html += '<a href="#" data-page="' + i + '">' + i + '</a>';
                }
            }

            // Dernière page
            if (data.current < data.total_pages) {
                html += '<a href="#" data-page="' + (data.current + 1) + '">&rsaquo;</a>';
                html += '<a href="#" data-page="' + data.total_pages + '">&raquo;</a>';
            }

            $links.html(html);
        },

        /**
         * Gérer le clic sur filtre
         */
        handleFilterClick: function(e) {
            e.preventDefault();
            
            var $link = $(e.currentTarget);
            this.currentFilter = $link.data('filter');
            this.currentPage = 1;
            
            $('.fliz-filter-link').removeClass('current');
            $link.addClass('current');
            
            this.loadPosts();
        },

        /**
         * Appliquer les filtres
         */
        applyFilters: function() {
            this.currentCategory = $('#fliz-filter-category').val();
            this.currentPage = 1;
            this.loadPosts();
        },

        /**
         * Toggle select all
         */
        toggleSelectAll: function(e) {
            var isChecked = $(e.currentTarget).prop('checked');
            
            $('#fliz-select-all, #fliz-select-all-bottom').prop('checked', isChecked);
            
            $('.fliz-post-checkbox:not(:disabled)').each(function() {
                var $checkbox = $(this);
                var postId = parseInt($checkbox.val(), 10);
                
                $checkbox.prop('checked', isChecked);
                $checkbox.closest('tr').toggleClass('selected', isChecked);
                
                if (isChecked) {
                    if (this.selectedPosts.indexOf(postId) === -1) {
                        this.selectedPosts.push(postId);
                    }
                } else {
                    var index = this.selectedPosts.indexOf(postId);
                    if (index > -1) {
                        this.selectedPosts.splice(index, 1);
                    }
                }
            }.bind(this));
            
            this.updateSelectedCount();
        },

        /**
         * Gérer la sélection d'un post
         */
        handlePostSelect: function(e) {
            var $checkbox = $(e.currentTarget);
            var postId = parseInt($checkbox.val(), 10);
            var isChecked = $checkbox.prop('checked');
            
            $checkbox.closest('tr').toggleClass('selected', isChecked);
            
            if (isChecked) {
                if (this.selectedPosts.indexOf(postId) === -1) {
                    this.selectedPosts.push(postId);
                }
            } else {
                var index = this.selectedPosts.indexOf(postId);
                if (index > -1) {
                    this.selectedPosts.splice(index, 1);
                }
            }
            
            this.updateSelectAllState();
            this.updateSelectedCount();
        },

        /**
         * Mettre à jour l'état du select all
         */
        updateSelectAllState: function() {
            var $checkboxes = $('.fliz-post-checkbox:not(:disabled)');
            var $checked = $checkboxes.filter(':checked');
            
            var allChecked = $checkboxes.length > 0 && $checkboxes.length === $checked.length;
            $('#fliz-select-all, #fliz-select-all-bottom').prop('checked', allChecked);
        },

        /**
         * Mettre à jour le compteur de sélection
         */
        updateSelectedCount: function() {
            var count = this.selectedPosts.length;
            $('#fliz-selected-count').text(count);
            $('#fliz-bulk-generate-btn').prop('disabled', count === 0);
        },

        /**
         * Gérer le clic sur pagination
         */
        handlePaginationClick: function(e) {
            e.preventDefault();
            var page = $(e.currentTarget).data('page');
            if (page) {
                this.currentPage = page;
                this.loadPosts();
            }
        },

        /**
         * Ouvrir le modal
         */
        openModal: function() {
            if (this.selectedPosts.length === 0) {
                alert(flizBulk.strings.noSelection);
                return;
            }

            var count = this.selectedPosts.length;
            $('#fliz-modal-count').text(count);
            $('#fliz-modal-credits').text(count);
            $('#fliz-bulk-modal').show();
        },

        /**
         * Fermer le modal
         */
        closeModal: function() {
            $('#fliz-bulk-modal').hide();
        },

        /**
         * Démarrer la génération en masse
         */
        startBulkGeneration: function() {
            var format = $('input[name="fliz_bulk_format"]:checked').val();
            
            this.closeModal();
            this.showProgress();

            $.ajax({
                url: flizBulk.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fliz_bulk_generate',
                    security: flizBulk.nonce,
                    post_ids: this.selectedPosts,
                    format: format,
                    category: 'article' // Toujours "article"
                },
                success: function(response) {
                    this.hideProgress();
                    
                    if (response.success) {
                        alert(response.data.message);
                        this.selectedPosts = [];
                        this.updateSelectedCount();
                        this.loadPosts();
                    } else {
                        alert(response.data.message || flizBulk.strings.error);
                    }
                }.bind(this),
                error: function() {
                    this.hideProgress();
                    alert(flizBulk.strings.error);
                }.bind(this)
            });
        },

        /**
         * Afficher la progression
         */
        showProgress: function() {
            var total = this.selectedPosts.length;
            $('#fliz-progress-total').text(total);
            $('#fliz-progress-current').text('0');
            $('#fliz-progress-fill').css('width', '0%');
            $('#fliz-progress-overlay').show();
        },

        /**
         * Masquer la progression
         */
        hideProgress: function() {
            $('#fliz-progress-overlay').hide();
        },

        /**
         * Échapper le HTML
         */
        escapeHtml: function(text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    // Initialiser quand le DOM est prêt
    $(document).ready(function() {
        FlizBulk.init();
    });

})(jQuery);
