<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       firehawkcrm.com
 * @since      1.0.0
 *
 * @package    Fcrm_Tributes
 * @subpackage Fcrm_Tributes/public
 * @author     Grant Bevan <grant@firehawk.digital>
 */

class Fcrm_Tributes_Public
{

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string    $plugin_name       The name of the plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct($plugin_name, $version)
    {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_styles()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in Fcrm_Tributes_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The Fcrm_Tributes_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */
        wp_enqueue_style($this->plugin_name . '-glidejs-core', plugin_dir_url(__FILE__) . 'js/glidejs/css/glide.core.min.css', array(), $this->version, 'all');
        wp_enqueue_style($this->plugin_name . '-glidejs-theme', plugin_dir_url(__FILE__) . 'js/glidejs/css/glide.theme.min.css', array(), $this->version, 'all');
        wp_enqueue_style( 'jquery-slick-nav', plugin_dir_url( __FILE__ ) . 'css/slicknav.css', array(), $this->version, 'all' );
        wp_enqueue_style( 'select2', plugin_dir_url( __FILE__ ) . 'css/select2.min.css', array(), $this->version, 'all' );
        wp_enqueue_style( 'add-to-calendar-button', plugin_dir_url( __FILE__ ) . 'css/atcb.min.css', array(), $this->version, 'all' );

        wp_enqueue_style($this->plugin_name . '-jquery-modal', plugin_dir_url(__FILE__) . 'css/jquery.modal.min.css', array(), $this->version, 'all');
        wp_enqueue_style($this->plugin_name.'-lightgallery-css', plugin_dir_url( __FILE__ ) . 'css/lightgallery.min.css', array(), null, 'all');
        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/fcrm-tributes-public.css', array(), $this->version, 'all');
        // wp_enqueue_style('fontawesome', 'https://use.fontawesome.com/releases/v5.15.2/css/all.css', array(), $this->version, 'all');
    }

    /**
     * Register the JavaScript for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in Fcrm_Tributes_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The Fcrm_Tributes_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */


        wp_enqueue_script($this->plugin_name . '-popperjs', 'https://unpkg.com/@popperjs/core@2', array( 'jquery' ), $this->version, false);
        wp_enqueue_script($this->plugin_name . '-tippyjs', 'https://unpkg.com/tippy.js@6', array( 'jquery' ), $this->version, false);
        wp_enqueue_script('fontawesome', 'https://kit.fontawesome.com/0b4429dff6.js', array(), $this->version, false);
        wp_enqueue_script("bootstrap", plugin_dir_url(__FILE__) . 'js/bootstrap.js', array( 'jquery' ), $this->version, false);
        wp_enqueue_script('shufflejs', plugin_dir_url( __FILE__ ) . 'js/shuffle.min.js', array( 'jquery' ), $this->version, false);
        wp_enqueue_script( 'jquery-history', plugin_dir_url( __FILE__ ) . 'js/jquery.history.js', array( 'jquery' ), $this->version, false);
        wp_enqueue_script( 'jquery-slick-nav', plugin_dir_url( __FILE__ ) . 'js/jquery.slicknav.min.js', array( 'jquery' ), $this->version, false);
        wp_enqueue_script( 'jquery-validate', plugin_dir_url( __FILE__ ) . 'js/jquery.validate.js', array( 'jquery' ), $this->version, false);
        wp_enqueue_script("select2", plugin_dir_url( __FILE__ ) . 'js/select2.full.js', array( 'jquery' ), $this->version, false);
        wp_enqueue_script( 'jquery-slick-carousel', plugin_dir_url( __FILE__ ) . 'js/slick.min.js', array( 'jquery' ), $this->version, false);

        wp_enqueue_script($this->plugin_name . '-clipboard', plugin_dir_url(__FILE__) . 'js/clipboard.min.js', array(), $this->version, false);
        wp_enqueue_script($this->plugin_name . '-textFit', plugin_dir_url(__FILE__) . 'js/textFit.min.js', array(), $this->version, false);
        wp_enqueue_script($this->plugin_name . '-glidejs', plugin_dir_url(__FILE__) . 'js/glidejs/glide.min.js', array( 'jquery' ), $this->version, false);
        wp_enqueue_script($this->plugin_name . '-jquery-modal', plugin_dir_url(__FILE__) . 'js/jquery.modal.min.js', array( 'jquery' ), $this->version, false);
        wp_enqueue_script($this->plugin_name . '-litepicker', plugin_dir_url(__FILE__) . 'js/litepicker.js', array( 'jquery' ), $this->version, false);

        wp_enqueue_script('add-to-calendar-button', plugin_dir_url(__FILE__) . 'js/atcb.js', array('jquery'), $this->version, false);

        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/sharer.min.js', array( 'jquery' ), $this->version, false);

        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/fcrm-tributes-public.js', array( 'jquery' ), $this->version, false);

        wp_enqueue_script("momentScript", plugin_dir_url(__FILE__) . 'js/moment.js', array(), null, false);
        wp_enqueue_script($this->plugin_name.'-lightgallerys', plugin_dir_url( __FILE__ ) . 'js/lightgallery.min.js', array(), $this->version, false);
        wp_enqueue_script($this->plugin_name . '-tribute-messages', plugin_dir_url(__FILE__) . 'js/fcrm-tributes-messages.js', array( 'jquery', 'shufflejs'), $this->version, false);
        wp_enqueue_script($this->plugin_name . '-tribute-trees', plugin_dir_url(__FILE__) . 'js/fcrm-tributes-trees.js', array( 'jquery', 'shufflejs'), $this->version, false);
        wp_enqueue_script($this->plugin_name . '-tribute-donations', plugin_dir_url(__FILE__) . 'js/fcrm-tributes-donations.js', array( 'jquery', 'shufflejs'), $this->version, false);
        wp_enqueue_script($this->plugin_name . '-tributes-grid', plugin_dir_url(__FILE__) . 'js/fcrm-tributes-grid.js', array( 'jquery'), $this->version, false);
        wp_enqueue_script($this->plugin_name . '-verify-input', plugin_dir_url(__FILE__) . 'js/fcrm-verify-input.js', array( 'jquery' ), $this->version, false);

    		wp_enqueue_script('lg-pager', plugin_dir_url( __FILE__ ) . 'js/lg-pager.min.js', array(), $this->version, false);
    		wp_enqueue_script('lg-zoom', plugin_dir_url( __FILE__ ) . 'js/lg-zoom.min.js', array(), $this->version, false);

        // wp_enqueue_script($this->plugin_name .'-flower-storefront', plugin_dir_url( __FILE__ ) . 'js/florist-one-flower-delivery-public.js', array( 'jquery', 'jquery-history' ), $this->version, false );
        wp_enqueue_script($this->plugin_name .'-flower-delivery', plugin_dir_url( __FILE__ ) . 'js/fcrm-tributes-flower-delivery.js', array( 'jquery', 'jquery-history' ), $this->version, false );

        wp_localize_script($this->plugin_name, 'ajax_var', array( 'url' => admin_url('admin-ajax.php'), 'nonce' => wp_create_nonce( "process_request_nonce" )));
    }

    private function format_client_name($client, $format) {
      $names = array();
      if ($format == "display") {
        if (isset($client->firstName)) array_push($names, $client->firstName);
        if (isset($client->otherNames)) array_push($names, $client->otherNames);
        if (isset($client->lastName)) array_push($names, $client->lastName);
        return join(" ", $names);
      } else if ($format == "display-full") {
        if (isset($client->firstName)) array_push($names, $client->firstName);
        if (isset($client->knownAs)) array_push($names, "\"".$client->knownAs."\"");
        if (isset($client->otherNames)) array_push($names, $client->otherNames);
        if (isset($client->lastName)) array_push($names, $client->lastName);
        if (isset($client->postNominalName)) array_push($names, $client->postNominalName);
        return join(" ", $names);
      } else if ($format == "display-details") {
        if (isset($client->firstName)) array_push($names, $client->firstName);
        if (isset($client->knownAs)) array_push($names, "\"".$client->knownAs."\"");
        if (isset($client->otherNames)) array_push($names, $client->otherNames);
        if (isset($client->lastName)) array_push($names, $client->lastName);
        return join(" ", $names);
      } else {
        if (isset($client->lastName)) array_push($names, $client->lastName.',');
        if (isset($client->firstName)) array_push($names, $client->firstName);
        if (isset($client->otherNames)) array_push($names, $client->otherNames);
        if (isset($client->knownAs)) array_push($names, "\"".$client->knownAs."\"");
        return join(" ", $names);
      }
    }

    private function clean_string($string) {
       $string = str_replace('-', '', $string); // Removes all hyphens.
       $string = preg_replace('/\s+/', '', $string); // Removes all spaces.
       $string = preg_replace('/[^A-Za-z0-9\-]/', '', $string); // Removes special chars.

       return strtolower(preg_replace('/-+/', '', $string)); // Replaces multiple hyphens with single one.
    }

    private function format_client_permalink($client, $detailPage)
    {
      if (!isset($detailPage) || !isset($client)) {
        return null;
      }

      $use_custom_link = get_option("fcrm_tributes_readable_permalinks");

      if (empty($use_custom_link) || $use_custom_link == false || $use_custom_link == 0) {
        $use_custom_link = false;
      } else {
        $use_custom_link = true;
      }

      $permalink = $detailPage.'/'.$client->id;
      $url_parts = array();

      if ($use_custom_link === true) {
        if (!isset($client->fileNumber)) {
          return null;
        }

        if (isset($client->firstName)) {
          array_push($url_parts, $this::clean_string($client->firstName));
        } else {
          array_push($url_parts, "1");
        }

        if (isset($client->lastName)) {
          array_push($url_parts, $this::clean_string($client->lastName));
        } else {
          array_push($url_parts, "1");
        }

        $fileNumber = str_replace('/', "%2F", $client->fileNumber);
        array_push($url_parts, $fileNumber);
        return home_url($detailPage.'/'.join("-", $url_parts));
      }

      $permalink = $detailPage.'/'.$client->id;
      return home_url($permalink);
    }

    private function find_page_client_number($permalink)
    {
      if (!isset($permalink)) {
        return null;
      }

      $url_parts = explode('-', $permalink);
      $page_id = "";

      foreach ($url_parts as $key => $value) {
          if ($key != 0 && $key != 1) {
             $seperator = $key > 2 ? "-" : "";
             $page_id = $page_id.$seperator.$value;
          }
      }

      return empty($page_id) ? null : $page_id;
    }

    /**
     * Public function to return grid display of tributes
     *
     * @since 1.0.0
     */
    public function shortcode_crm_tributes_large_grid_display($atts)
    {
        $attributes = shortcode_atts(array(
          'name-format' => null,
          'detail-page' => null,
          'layout' => 'basic',
          'limit' => 6,
          'image-style' => 'basic',
          'click-action' => 'open',
          'from-today' => null,
          'range-months' => null,
          'range-days' => null,
          'range-hours' => null
        ), $atts);

        if (isset($attributes["limit"]) == false) {
          $attributes["limit"] = 6;
        } else if (isset($attributes["limit"]) && isset($attributes["limit"]) > 12) {
          $attributes["limit"] = 12;
        }

        $pageSize = $attributes["limit"];

        $detailPage = $attributes['detail-page'];
        $tributePage = get_option('fcrm_tributes_single_page');
        if (isset($tributePage) && !isset($detailPage)) {
          $detailPage = $tributePage;
        }
        $clickAction = $attributes['click-action'];

        $layout = $attributes['layout'];
        $imageStyle = $attributes['image-style'];

        if (isset($layout) == false) {
          $layout = "basic";
        }

        if (isset($imageStyle) == false) {
          $imageStyle = "basic";
        }

        $searchRange = [];
        if (isset($attributes["from-today"])) {
          $fromToday = $attributes["from-today"] === "true";
        }
        if (isset($attributes["range-months"]) && $attributes["range-months"] > 0) {
          $searchRange["months"] = $attributes["range-months"];
        }
        if (isset($attributes["range-days"]) && $attributes["range-days"] > 0) {
          $searchRange["days"] = $attributes["range-days"];
        }

        if (isset($attributes["range-hours"]) && $attributes["range-hours"] > 0) {
          $searchRange["hours"] = $attributes["range-hours"];
        }

        $attributes["sort"] = [[
    			"field" => "serviceDateTime",
    			"order" => "desc"
    		]];

        $clients = Fcrm_Tributes_Api::get_clients($attributes);
        $clients = isset($clients) ? $clients : [];

        for ($i = 0; $i < count($clients); $i++) {
            $client = $clients[$i];
            $client->fullName = $this::format_client_name($client, $attributes["name-format"]);
            $client->permalink = $this::format_client_permalink($client, $detailPage);
        }

        ob_start();
        require_once (plugin_dir_path(dirname(__FILE__)) . 'public/partials/fcrm-tributes-public-large-grid-display.php');
        $content = ob_get_clean();
        return $content;
    }

    public function shortcode_crm_tributes_carousel_display($atts) {

      $attributes = shortcode_atts(array(
        'name-format' => null,
        'detail-page' => null,
        'layout' => 'basic',
        'size' => 12
      ), $atts);

      $team = get_option('fcrm_team');
      $size = 12;

      if(isset($attributes['size'])) {
        $size = intval($attributes['size']);
      }

      $detailPage = $attributes['detail-page'];
      $tributePage = get_option('fcrm_tributes_single_page');
      if (isset($tributePage) && !isset($detailPage)) {
        $detailPage = $tributePage;
      }

      $response = Fcrm_Tributes_Api::get_tributes([
        "limit" => $size
      ]);

      $clients = [];
      if (isset($response->results)) {
        $clients = $response->results;
        for ($i = 0; $i < count($clients); $i++) {
            $client = $clients[$i];
            $client->fullName = $this::format_client_name($client, null);
            $client->permalink = $this::format_client_permalink($client, $detailPage);
        }
      }

      ob_start();
      require_once(plugin_dir_path(dirname(__FILE__)) . 'public/partials/fcrm-tributes-public-carousel-display.php');
      $content = ob_get_clean();
      return $content;
   }

    public function shortcode_crm_tributes_grid_display($attributes) {
  		$team = get_option('fcrm_team');
  		$size = 12;
      $sortByService = false;
      $nameFormat = null;
      $displayServiceInfo = false;
      $hideDateOfBirth = false;
      $dateFormat = "";

  		if (isset($attributes['size'])) {
  			$size = intval($attributes['size']);
  		}

      if (isset($attributes['name-format'])) {
  			$nameFormat = $attributes['name-format'];
  		}

      if (isset($attributes['date-format'])) {
  			$dateFormat = $attributes['date-format'];
  		}

      if (get_option('fcrm_tributes_hide_dob') == true) {
        $hideDateOfBirth = true;
      } else if (isset($attributes['hide-dob'])) {
        $hideDateOfBirth = ($attributes['hide-dob'] === true || $attributes['hide-dob'] === "true") ? true : false;
      }

      

      if (isset($attributes['display-service'])) {
        $displayServiceInfo = $attributes['display-service'] == true || $attributes['display-service'] == "true";
      }

      if (isset($attributes['sort-by-service'])) {
  			$sortByService = $attributes['sort-by-service'] == true || $attributes['sort-by-service'] == "true";
  		}

  		ob_start();
      require_once(plugin_dir_path(dirname(__FILE__)) . 'public/partials/fcrm-tributes-public-grid-display.php');
      $content = ob_get_clean();
      return $content;
	 }

   public function shortcode_crm_tributes_flower_delivery() {
     return Fcrm_Tributes_Flowers_Api::init_flower_delivery();
   }

    /**
     * Public function to return grid display of tributes
     *
     * @since 1.0.0
     */
    public function large_grid_fetch_more()
    {
        $from = $_POST['from'];
        $size = $_POST['size'];
        $layout = $_POST['layout'];
        $imageStyle = $_POST['imageStyle'];
        $detailPage = $_POST['detailsPage'];
        $range = $_POST['range'];
        $clickAction = $_POST['click-action'];

        if (isset($size) == false || $size == 0) {
          $size = 20;
        }

        if (isset($layout) == false) {
          $layout = "basic";
        }

        if (isset($clickAction) == false) {
          $clickAction = "open";
        }

        $attributes = [
          "limit" => $size,
          "from" => $from,
          "sort" => [[
            "field" => "serviceDateTime",
            "order" => "desc"
          ]]
        ];

        if (isset($_POST["from-today"])) {
          $fromToday = $_POST["from-today"] === "true";
        }
        if (isset($_POST["range-months"]) && $_POST["range-months"] > 0) {
          $attributes["range-months"] = $_POST["range-months"];
        }
        if (isset($_POST["range-days"]) && $_POST["range-days"] > 0) {
          $attributes["range-days"] = $_POST["range-days"];
        }

        if (isset($_POST["range-hours"]) && $_POST["range-hours"] > 0) {
          $attributes["range-hours"] = $_POST["range-hours"];
        }

        $clients = Fcrm_Tributes_Api::get_clients($attributes);
        $clients = isset($clients) ? $clients : [];

        if (count($clients) <= 0) {
          echo "";
          die();
        }
        for ($i = 0; $i < count($clients); $i++) {
            $client = $clients[$i];
            $client->fullName = $this::format_client_name($client, $attributes["name-format"]);
            $client->permalink = $this::format_client_permalink($client, $detailPage);
        }

        ob_start();
        require_once (plugin_dir_path(dirname(__FILE__)) . 'public/partials/fcrm-tributes-public-large-grid-display-more.php');
        $content = ob_get_clean();
        echo $content;
        die();
    }

    /**
     * Public function to return grid display of tributes
     *
     * @since 1.0.0
     */
    public function get_tributes()
    {
        $params = $_POST['params'];
        if (!isset($params)) {
          $params = [];
        }
        $from = $params['from'];
        $size = $params['size'];
        $query = $params['query'];
        $nameFormat = null;

        if (isset($params["nameFormat"])) {
          $nameFormat = $params["nameFormat"];
        }

        $startDate = $params['startDate'];
        $endDate = $params['endDate'];

        if (isset($size) == false || $size == 0) {
          $size = 20;
        }

        $detailPage = $params['detail-page'];
        $tributePage = get_option('fcrm_tributes_single_page');
        if (isset($tributePage) && !isset($detailPage)) {
          $detailPage = $tributePage;
        }

        $attributes = [
          "limit" => $size,
          "from" => $from,
          "startDate" => $startDate,
          "endDate" => $endDate,
        ];

        if (isset($params["sortByService"]) && $params["sortByService"] == true) {
          $attributes["sort"] = [
            ["field" => "serviceDateTime", "order" => "desc"],
            ["field" => "clientDateOfDeath", "order" => "desc"]
          ];
        }

        if (isset($params["displayService"]) && $params["displayService"] == true) {
          $attributes["displayService"] = true;
        }

        if (isset($query)) {
          $attributes["fuzzy"] = array(
        		'query' => $query,
        		'fields' => array(
        			'firstName',
        			'lastName',
        			'name'
        		),
        		'condition' => 'must',
        		'type' => 'phrase_prefix',
        	);
        }

        $response = Fcrm_Tributes_Api::get_tributes($attributes);
        if (!isset($response) || !isset($response->results)) {
          echo json_encode([ "results" => [], "total" => 0]);
          die();
        }

        for ($i = 0; $i < count($response->results); $i++) {
            $client = $response->results[$i];
            $client->fullName = $this::format_client_name($client, $nameFormat);
            $client->permalink = $this::format_client_permalink($client, $detailPage);
        }

        echo json_encode($response);
        die();
    }

    public function get_tribute_messages()
    {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::get_tribute_messages($params);
        echo json_encode($response);
        die();
    }

    public function get_tribute_trees()
    {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::get_tribute_trees($params);
        echo json_encode($response);
        die();
    }

    public function get_tribute_donations()
    {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::get_tribute_donations($params);
        echo json_encode($response);
        die();
    }

    public function verify_tribute_code() {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::verify_tribute_code($params);
        echo json_encode($response);
        die();
    }

    public function verify_tribute_email() {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::verify_tribute_email($params);
        echo json_encode($response);
        die();
    }

    public function verify_edit_tribute_email() {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::verify_edit_tribute_email($params);
        echo json_encode($response);
        die();
    }

    public function like_tribute_message() {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::like_tribute_message($params);
        echo json_encode($response);
        die();
    }

    public function unlike_tribute_message() {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::unlike_tribute_message($params);
        echo json_encode($response);
        die();
    }

    public function like_tribute_tree() {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::like_tribute_tree($params);
        echo json_encode($response);
        die();
    }

    public function unlike_tribute_tree() {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::unlike_tribute_tree($params);
        echo json_encode($response);
        die();
    }

    public function like_tribute_donation() {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::like_tribute_donation($params);
        echo json_encode($response);
        die();
    }

    public function unlike_tribute_donation() {
        $params = array();
        foreach ($_POST['params'] as $key => $value) {
          $params[$key] = $value;
        }

        $response = Fcrm_Tributes_Api::unlike_tribute_donation($params);
        echo json_encode($response);
        die();
    }

    public function create_tribute_message() {
        $params = array();
        foreach ($_POST as $key => $value) {
          if (is_string($value)) {
            $params[$key] = stripslashes($value);
          } else {
            $params[$key] = $value;
          }
        }
        $response = Fcrm_Tributes_Api::create_tribute_message($params);
        echo json_encode($response);
        die();
    }

    public function edit_tribute_message() {
        $params = array();
        foreach ($_POST as $key => $value) {
          if (is_string($value)) {
            $params[$key] = stripslashes($value);
          } else {
            $params[$key] = $value;
          }
        }

        $response = Fcrm_Tributes_Api::edit_tribute_message($params);
        echo json_encode($response);
        die();
    }

    public function create_tribute_tree() {
        $params = array();
        foreach ($_POST as $key => $value) {
          if (is_string($value)) {
            $params[$key] = stripslashes($value);
          } else {
            $params[$key] = $value;
          }
        }
        $response = Fcrm_Tributes_Api::create_tribute_tree($params);
        echo json_encode($response);
        die();
    }

    public function create_tribute_donation() {
        $params = array();
        foreach ($_POST as $key => $value) {
          if (is_string($value)) {
            $params[$key] = stripslashes($value);
          } else {
            $params[$key] = $value;
          }
        }
        $response = Fcrm_Tributes_Api::create_tribute_donation($params);
        echo json_encode($response);
        die();
    }

    public function ajax_add_tree_to_cart() {
      $quantity = (int)$_POST["quantity"];
      $contact = $_POST["contact"];

      if (!isset($quantity) || $quantity < 1) {
        $quantity = 1;
      }

      $params = [
        "disableCookie" => true,
        "createInvoiceOrder" => true,
        "paymentTeam" => "-MHtwQZjrJI5morFgGKc",
        "noShipping" => true,
        "contact" => $contact,
        "items" => [
          [
          "quantity" => $quantity,
          "stock" => "-N9ZoZ3tpHDOXS3FQkhg"
          ]
        ]
      ];

      if (class_exists("Firehawkcrm_Cart_Public") == false) {
        wp_send_json_error("Firehawk Cart not configured", 400);
      }

      $response = Firehawkcrm_Cart_Public::replace_cart($params);

      if (!isset($response) || !isset($response['result'])) {
			    wp_send_json_error("Unknown error", 400);
		  }

		  wp_send_json($response);
    }

    public function ajax_add_donation_to_cart() {
      $amount = (float)$_POST["amount"];
      $contact = $_POST["contact"];
      $currency = $_POST["currency"];

      if (!isset($amount)) {
        wp_send_json_error("Amount is not valid", 400);
      }

      $params = [
        "disableCookie" => true,
        "createInvoiceOrder" => true,
        "paymentTeam" => "-MHtwQZjrJI5morFgGKc",
        "noShipping" => true,
        "contact" => $contact,
        "items" => [
          [
          "stock" => "-N9jqQFSEzTfpdUZw7ZG",
          "quantity" => 1,
          "priceValues" => [
            "priceExclTax" => $amount,
            "priceInclTax" => $amount
            ]
          ]
        ]
      ];

      if (isset($currency)) {
        $params["currency"] = $currency;
      }

      if (class_exists("Firehawkcrm_Cart_Public") == false) {
        wp_send_json_error("Firehawk Cart not configured", 400);
      }

      $response = Firehawkcrm_Cart_Public::replace_cart($params);

      if (!isset($response) || !isset($response['result'])) {
          wp_send_json_error("Unknown error", 400);
      }

      wp_send_json($response);
    }

    /**
     * Public function to return details of a service
     *
     * @since 1.0.0
     */
    public function shortcode_crm_tribute_display($atts)
    {
        $attributes = shortcode_atts(array(
          'name-format' => null,
          'detail-page' => null,
          'layout' => 'basic',
          'heading-layout' => 'basic'
        ), $atts);

        $team = get_option('fcrm_tributes_team');
        $detailPage = $attributes['detail-page'];
        $tributePage = get_option('fcrm_tributes_single_page');
        if (isset($tributePage) && !isset($detailPage)) {
          $detailPage = $tributePage;
        }
        $eventDateFormat = get_option('fcrm_tributes_event_date_format');
        $eventEndDateFormat = get_option('fcrm_tributes_event_end_date_format');

        $single_tribute = new Single_Tribute();
        $single_tribute->detectClient();

        $client = $single_tribute->getClient();

        if (!isset($client) || isset($client->error) || $client->displayTributes != true) {
          return $this::error_view("Tribute not found", "We couldn't find this tribute.");
        } else {
          $client->fullName = $this::format_client_name($client, $attributes["name-format"]);
          $client->permalink = $this::format_client_permalink($client, $detailPage);
        }

        $layout = $attributes['layout'];
        $headingLayout = $attributes['heading-layout'];

        $canProcessPayments = class_exists("Firehawkcrm_Cart_Public");
        $canPlantTrees = $client->displayTributeTrees === true && $canProcessPayments;
        $canMakeDonations = $client->displayTributeDonations === true && $canProcessPayments && isset($client->donationsCharity);
        $carouselGallery = true;
        $can_deliver_flowers = false;

        $checkoutCartUrl = null;

        if (class_exists("Firehawkcrm_Cart_Public")) {
          $checkoutCartUrl = Firehawkcrm_Cart_Public::getCheckoutSlug();
          if ($checkoutCartUrl) $checkoutCartUrl = home_url($checkoutCartUrl);
        }

        if ($client->displayTributeFlowers == false) {
          $can_deliver_flowers = false;
        }

        if (isset($layout) == false) {
          $layout = "basic";
        }

        $showMenuTabs = true;
        $showTabs = array();

        if ($client->displayTributeMessages === true) {
          $showTabs[] = "messages";
        }

        if ($can_deliver_flowers === true) {
          $showTabs[] = "flowers";
        }

        if ($canPlantTrees) {
          $showTabs[] = "trees";
        }

        if ($canMakeDonations) {
          $showTabs[] = "donations";
        }

        if ($client->displayLiveStream === true && $client->liveStreamEmbedUrl) {
          $showTabs[] = "live-stream";
        } else if (isset($client->liveStreamUrl)) {
          $showTabs[] = "live-stream";
        }

        if (isset($client->fragmentTributeUrl)) {
          $showTabs[] = "social-tribute";
        }

        if (isset($client->content)) {
          if ($client->content != strip_tags($client->content)) {
            $client->formatted_content = $client->content;
          } else {
            $client->formatted_content = nl2br($client->content);
          }
        }

        ob_start();
        require_once (plugin_dir_path(dirname(__FILE__)) . 'public/partials/tributes/single-tribute-display.php');
        $content = ob_get_clean();
        return $content;
    }

    public function ajax_get_tree_stock() {
      $treeStock = Fcrm_Tributes_Api::get_stock("-N9ZoZ3tpHDOXS3FQkhg", "-MHtwQZjrJI5morFgGKc");

      if (isset($treeStock)) {
        wp_send_json($treeStock);
      } else {
        wp_send_json_error("Stock not found", 400);
      }
    }

    public function ajax_get_client_donation_info() {
      $clientId = $_POST["clientId"];
      $info = Fcrm_Tributes_Api::get_client_donation_info($clientId);

      if (isset($info)) {
        wp_send_json($info);
      } else {
        wp_send_json_error("Charity not found", 400);
      }
    }

    function getRelativeDate($date) {
        $today = new DateTime(); // This object represents current date/time
        $today->setTime( 0, 0, 0 ); // reset time part, to prevent partial comparison

        $match_date = new DateTime($date);
        $match_date->setTime( 0, 0, 0 ); // reset time part, to prevent partial comparison

        $diff = $today->diff( $match_date );
        $diffDays = (integer)$diff->format( "%R%a" ); // Extract days count in interval

        switch( $diffDays ) {
            case 0:
                return "Today at";
                break;
            case -1:
                return "Yesterday at";
                break;
            case +1:
                return "Tomorrow at";
                break;
            default:
                return null;
        }
    }

    function tribute_details_meta( $posts ) {
      $shortcode = 'show_crm_tribute';
      $callback_function = 'service_details_setmeta';

      return $this::service_details_to_wphead( $posts, $shortcode, $callback_function );
    }

    // To execute when shortcode is found
    function service_details_setmeta() {

      // global $client;
      global $wp;
      $meta = '';
      $current_url = home_url(add_query_arg(array($_GET), $wp->request));
      $use_custom_link = get_option("fcrm_tributes_readable_permalinks");

      if (empty($use_custom_link) || $use_custom_link == false || $use_custom_link == 0) {
        $use_custom_link = false;
      } else {
        $use_custom_link = true;
      }

      $id = $wp->query_vars["id"];
      $page_id = $this::find_page_client_number($id);
      $client = null;

      if (isset($page_id) && $use_custom_link == true) {
        $client = Fcrm_Tributes_Api::get_client_by_number($page_id, false, true);
      } else {
        $client = Fcrm_Tributes_Api::get_client($id, false, true);
      }

      if (isset($client)) {
        $fcrmDefaultImageUrl = get_option('fcrm_tributes_default_image');
        $clientName = $this::format_client_name($client, "display");
        $client->name = $clientName;

        if (isset($client->clientDateOfBirth)) {
          $clientDateOfBirth = new DateTime($client->clientDateOfBirth);
          $client->yearOfBirth = $clientDateOfBirth->format("Y");
          $client->dateOfBirth = $clientDateOfBirth->format("F jS Y");
        }

        if (isset($client->clientDateOfDeath)) {
          $clientDateOfDeath = new DateTime($client->clientDateOfDeath);
          $client->yearOfDeath = $clientDateOfDeath->format("Y");
          $client->dateOfDeath = $clientDateOfDeath->format("F jS Y");
        }
        
        $metaTitle = $clientName.' - Online Tribute';

        $decodedClient = json_decode(json_encode($client), true);

        $customMetaTitle = rtrim(get_option('fcrm_tributes_meta_title', ""));
        $customMetaDescription = rtrim(get_option('fcrm_tributes_meta_description', ""));
        
        if (empty($customMetaTitle) == false) {
          $metaTitle = $this::replace_tags($customMetaTitle, $decodedClient);
        }

        $meta = $meta . '<meta property="og:url" content="'.$current_url.'">';
        $meta = $meta . '<meta property="og:url:secure_url" content="'.$current_url.'">';
        $meta = $meta . '<meta property="og:type" content="article">';
        $meta = $meta . '<meta property="og:title" content="'.$metaTitle.'">';
        $meta = $meta . '<meta property="title" content="'.$metaTitle.'">';

        if (empty($customMetaDescription) == false) {
          $metaDescription = $this::replace_tags($customMetaDescription, $decodedClient);
          $meta = $meta . '<meta property="og:description" content="'.$metaDescription.'">';
          $meta = $meta . '<meta property="description" content="'.$metaDescription.'">';
        } else if (isset($client->content)) {
          if ($client->content != strip_tags($client->content)) {
            $meta = $meta . '<meta property="og:description" content="'.strip_tags($client->content).'">';
            $meta = $meta . '<meta property="description" content="'.strip_tags($client->content).'">';
          } else {
            $meta = $meta . '<meta property="og:description" content="'.$client->content.'">';
            $meta = $meta . '<meta property="description" content="'.$client->content.'">';
          }
        }

        if (isset($client->displayImage)) {
          $meta = $meta . '<meta property="og:image" content="'.$client->displayImage.'">';
          $meta = $meta . '<meta property="og:image:secure_url" content="'.$client->displayImage.'">';
          $meta = $meta . '<meta itemprop="image" content="'.$client->displayImage.'">';
        } elseif (isset($client->bannerImage)) {
          $meta = $meta . '<meta property="og:image" content="'.$client->bannerImage.'">';
          $meta = $meta . '<meta property="og:image:secure_url" content="'.$client->bannerImage.'">';
          $meta = $meta . '<meta itemprop="image" content="'.$client->bannerImage.'">';
        } elseif ($fcrmDefaultImageUrl && strlen($fcrmDefaultImageUrl)) {
          $meta = $meta . '<meta property="og:image" content="'.$fcrmDefaultImageUrl.'">';
          $meta = $meta . '<meta property="og:image:secure_url" content="'.$fcrmDefaultImageUrl.'">';
          $meta = $meta . '<meta itemprop="image" content="'.$fcrmDefaultImageUrl.'">';
        }
      }
      echo $meta;
    }

    // look for shortcode in the content and apply expected behaviour (don't edit!)
    function service_details_to_wphead( $posts, $shortcode, $callback_function ) {
      if ( empty( $posts ) )
        return $posts;

      $found = false;
      foreach ( $posts as $post ) {
        if ( stripos( $post->post_content, '[' . $shortcode ) !== false ) {
          $found = true;
          break;
        }
      }

      if ( $found ) {
        global $wpseo_front;
        if(defined($wpseo_front)){
          remove_action('wp_head',array($wpseo_front,'head'),1);
        } else if (class_exists('WPSEO_Frontend')) {
          $wp_thing = WPSEO_Frontend::get_instance();
          remove_action('wp_head',array($wp_thing,'head'),1);
        }
        add_action( 'wp_head', array($this, $callback_function) );
      }

      return $posts;
    }

    // Update the canonical URL for single tributes for the Yoast SEO plugin
    function update_yoast_single_tribute_canonical_url( $canonical ) {
      
      if ( is_page( Single_Tribute::getSingleTributePageId() ) ) {
        $single_tribute = new Single_Tribute();
        $single_tribute->detectClient();
        $canonical = $single_tribute->getPageUrl();
      }
    
      return $canonical;
    }

    function update_yoast_single_tribute_opengraph_url( $opengraph_url ) {
      
      if ( is_page( Single_Tribute::getSingleTributePageId() ) ) {
        $single_tribute = new Single_Tribute();
        $single_tribute->detectClient();
        $opengraph_url = $single_tribute->getPageUrl();
      }
    
      return $opengraph_url;
    }

    function update_yoast_single_tribute_opengraph_title( $opengraph_title ) {
      
      if ( is_page( Single_Tribute::getSingleTributePageId() ) ) {
        $single_tribute = new Single_Tribute();
        $single_tribute->detectClient();
        $client = $single_tribute->getClient();
        $clientName = $this::format_client_name($client, "display");
        $metaTitle = $clientName.' - Online Tribute';
        $customMetaTitle = rtrim(get_option('fcrm_tributes_meta_title', ""));
        if (empty($customMetaTitle) == false) {
          $metaTitle = $this::replace_tags($customMetaTitle, $decodedClient);
        }
        $opengraph_title = $metaTitle;
      }
    
      return $opengraph_title;
    }

    function update_yoast_single_tribute_opengraph_image( $opengraph_image ) {
      
      if ( is_page( Single_Tribute::getSingleTributePageId() ) ) {
        $single_tribute = new Single_Tribute();
        $single_tribute->detectClient();
        $client = $single_tribute->getClient();
        if (isset($client->displayImage)) {
          $opengraph_image = $client->displayImage;
        } elseif (isset($client->bannerImage)) {
          $opengraph_image = $client->bannerImage;
        } elseif ($fcrmDefaultImageUrl && strlen($fcrmDefaultImageUrl)) {
          $opengraph_image = $fcrmDefaultImageUrl;
        }
      }
    
      return $opengraph_image;
    }

    function update_yoast_single_tribute_opengraph_desc( $opengraph_desc ) {
      
      if ( is_page( Single_Tribute::getSingleTributePageId() ) ) {
        $single_tribute = new Single_Tribute();
        $single_tribute->detectClient();
        $client = $single_tribute->getClient();
        $clientName = $this::format_client_name($client, "display");
        $customMetaDescription = rtrim(get_option('fcrm_tributes_meta_description', ""));
        if (empty($customMetaDescription) == false) {
          $opengraph_desc = $this::replace_tags($customMetaDescription, $decodedClient);
        } else if (isset($client->content)) {
          if ($client->content != strip_tags($client->content)) {
            $opengraph_desc = strip_tags($client->content);
          } else {
            $opengraph_desc = $client->content;
          }
        }
      }
    
      return $opengraph_desc;
    }

    function yoast_add_to_sitemap_index($sitemap_custom_items) {
      $sitemap_custom_items .= '
        <sitemap>
          <loc>'.get_site_url().'/fhf_tributes_sitemap_1.xml</loc>
          <lastmod>'.date("Y-m-d\TH:i:sP").'</lastmod>
        </sitemap>';
      return $sitemap_custom_items;
    }

    // function initiate_generate_tribute_sitemap() {
    //   if(!wp_next_scheduled('firehawk_tribute_sitemap_generate_hourly')) {
    //     wp_schedule_event(current_time('timestamp'), 'hourly', 'firehawk_tribute_sitemap_generate_hourly');
    //   }
    // }

    // function generate_tribute_sitemap() {
    //   $client_data = Fcrm_Tributes_Api::get_tributes_sitemap();
    //   if(isset($client_data) && isset($client_data->results)) {
    //     foreach($client_data->results as $client) {
    //       // var_dump($client);
    //       // echo $client->name;
    //       $tributePage = get_option('fcrm_tributes_single_page');
    //       $permalink = $this::format_client_permalink($client, $tributePage);
    //       echo $permalink;
    //     }
    //   }
    //   // var_dump($client_data);
    // }

    function tribute_sitemap_template_include($template)
    {
        global $wp_query; //Load $wp_query object
        $page_value = $wp_query->query_vars['fhf_tributes_sitemap']; //Check for query var "fhf_tributes_sitemap"
        if ($page_value) { //Verify "fhf_tributes_sitemap" exists
            $page_num = (int)$page_value;
            if($page_num) {
              return plugin_dir_path(__FILE__).'partials/fcrm-tributes-sitemap.php'; //Load your template or file
            }
        }

        return $template; //Load normal template when $page_value != "true" as a fallback
    }

    function tribute_sitemap_redirect_canonical_callback( $redirect_url, $requested_url ) {
        $is_load_fhf_tribute_sitemap = (bool) get_query_var( 'fhf_tributes_sitemap' );
    
        if ( $is_load_fhf_tribute_sitemap ) {
            return $requested_url;
        }
    
        return $redirect_url;
    }

    function plugin_query_vars( $qvars ) {
        $qvars[] = 'id';
        return $qvars;
    }

    function error_view($title, $message) {
      $error_message = $message;
      $error_message = "<h4>". $title ."</h4><p>". $message ."</p>";
      ob_start();
      require_once(plugin_dir_path(dirname(__FILE__)) . 'public/partials/fcrm-tributes-public-error.php');
      $content = ob_get_clean();
      return $content;
    }

    function replace_tags($string, $tags, $force_lower = false) {
      return preg_replace_callback('/\\{\\{([^{}]+)\}\\}/',
        function($matches) use ($force_lower, $tags)
        {
          $key = $force_lower ? strtolower($matches[1]) : $matches[1];
          return array_key_exists($key, $tags) 
            ? $tags[$key] 
            : ''
            ;
        }
        , $string);
    }
}
