<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       firehawkcrm.com
 * @since      1.0.0
 *
 * @package    Fcrm_Tributes
 * @subpackage Fcrm_Tributes/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Fcrm_Tributes
 * @subpackage Fcrm_Tributes/includes
 * @author     Grant Bevan <grant@firehawk.digital>
 */
class Fcrm_Tributes {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Fcrm_Tributes_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'PLUGIN_NAME_VERSION' ) ) {
			$this->version = PLUGIN_NAME_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'fcrm-tributes';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Fcrm_Tributes_Loader. Orchestrates the hooks of the plugin.
	 * - Fcrm_Tributes_i18n. Defines internationalization functionality.
	 * - Fcrm_Tributes_Admin. Defines all hooks for the admin area.
	 * - Fcrm_Tributes_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-fcrm-tributes-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-fcrm-tributes-i18n.php';

		/**
		 * BTF API functions
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/fcrm-api.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-fcrm-tributes-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-single-tribute-type.php';

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-fcrm-tributes-public.php';

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-fcrm-tributes-flower-delivery.php';

		$this->loader = new Fcrm_Tributes_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Fcrm_Tributes_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Fcrm_Tributes_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Fcrm_Tributes_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_admin_menu' );
		$this->loader->add_action( 'admin_init', $plugin_admin, 'settings_api_init' );

		$this->loader->add_action( 'add_option_fcrm_tributes_single_page', $plugin_admin, 'on_single_page_setting_update' );
		$this->loader->add_action( 'update_option_fcrm_tributes_single_page', $plugin_admin, 'on_single_page_setting_update' );
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Fcrm_Tributes_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

		$this->loader->add_action( 'query_vars', $plugin_public, 'plugin_query_vars' );
		$this->loader->add_action( 'the_posts', $plugin_public, 'tribute_details_meta' );

		$this->loader->add_action('wp_ajax_large_grid_fetch_more', $plugin_public, 'large_grid_fetch_more');
		$this->loader->add_action('wp_ajax_nopriv_large_grid_fetch_more', $plugin_public, 'large_grid_fetch_more');
		$this->loader->add_action('wp_ajax_get_tribute_messages', $plugin_public, 'get_tribute_messages');
		$this->loader->add_action('wp_ajax_nopriv_get_tribute_messages', $plugin_public, 'get_tribute_messages');

		$this->loader->add_action('wp_ajax_get_tributes', $plugin_public, 'get_tributes');
		$this->loader->add_action('wp_ajax_nopriv_get_tributes', $plugin_public, 'get_tributes');

		$this->loader->add_action('wp_ajax_verify_tribute_email', $plugin_public, 'verify_tribute_email');
		$this->loader->add_action('wp_ajax_nopriv_verify_tribute_email', $plugin_public, 'verify_tribute_email');

		$this->loader->add_action('wp_ajax_verify_tribute_code', $plugin_public, 'verify_tribute_code');
		$this->loader->add_action('wp_ajax_nopriv_verify_tribute_code', $plugin_public, 'verify_tribute_code');

		$this->loader->add_action('wp_ajax_verify_edit_tribute_email', $plugin_public, 'verify_edit_tribute_email');
		$this->loader->add_action('wp_ajax_nopriv_verify_edit_tribute_email', $plugin_public, 'verify_edit_tribute_email');

		$this->loader->add_action('wp_ajax_create_tribute_message', $plugin_public, 'create_tribute_message');
		$this->loader->add_action('wp_ajax_nopriv_create_tribute_message', $plugin_public, 'create_tribute_message');

		$this->loader->add_action('wp_ajax_edit_tribute_message', $plugin_public, 'edit_tribute_message');
		$this->loader->add_action('wp_ajax_nopriv_edit_tribute_message', $plugin_public, 'edit_tribute_message');

		$this->loader->add_action('wp_ajax_create_tribute_tree', $plugin_public, 'create_tribute_tree');
		$this->loader->add_action('wp_ajax_nopriv_create_tribute_tree', $plugin_public, 'create_tribute_tree');

		// Tribute Message Likes
		$this->loader->add_action('wp_ajax_like_tribute_message', $plugin_public, 'like_tribute_message');
		$this->loader->add_action('wp_ajax_nopriv_like_tribute_message', $plugin_public, 'like_tribute_message');

		$this->loader->add_action('wp_ajax_unlike_tribute_message', $plugin_public, 'unlike_tribute_message');
		$this->loader->add_action('wp_ajax_nopriv_unlike_tribute_message', $plugin_public, 'unlike_tribute_message');

		// Tribute Trees
		$this->loader->add_action('wp_ajax_like_tribute_tree', $plugin_public, 'like_tribute_tree');
		$this->loader->add_action('wp_ajax_nopriv_like_tribute_tree', $plugin_public, 'like_tribute_tree');

		$this->loader->add_action('wp_ajax_unlike_tribute_tree', $plugin_public, 'unlike_tribute_tree');
		$this->loader->add_action('wp_ajax_nopriv_unlike_tribute_tree', $plugin_public, 'unlike_tribute_tree');

		$this->loader->add_action('wp_ajax_get_tribute_trees', $plugin_public, 'get_tribute_trees');
		$this->loader->add_action('wp_ajax_nopriv_get_tribute_trees', $plugin_public, 'get_tribute_trees');

		$this->loader->add_action('wp_ajax_add_tree_to_cart', $plugin_public, 'ajax_add_tree_to_cart');
		$this->loader->add_action('wp_ajax_nopriv_add_tree_to_cart', $plugin_public, 'ajax_add_tree_to_cart');

		$this->loader->add_action('wp_ajax_get_tree_stock', $plugin_public, 'ajax_get_tree_stock');
		$this->loader->add_action('wp_ajax_nopriv_get_tree_stock', $plugin_public, 'ajax_get_tree_stock');

		$this->loader->add_action('wp_ajax_get_client_donation_info', $plugin_public, 'ajax_get_client_donation_info');
		$this->loader->add_action('wp_ajax_nopriv_get_client_donation_info', $plugin_public, 'ajax_get_client_donation_info');

		// Tribute Donations
		$this->loader->add_action('wp_ajax_create_tribute_donation', $plugin_public, 'create_tribute_donation');
		$this->loader->add_action('wp_ajax_nopriv_create_tribute_donation', $plugin_public, 'create_tribute_donation');


		$this->loader->add_action('wp_ajax_like_tribute_donation', $plugin_public, 'like_tribute_donation');
		$this->loader->add_action('wp_ajax_nopriv_like_tribute_donation', $plugin_public, 'like_tribute_donation');

		$this->loader->add_action('wp_ajax_unlike_tribute_donation', $plugin_public, 'unlike_tribute_donation');
		$this->loader->add_action('wp_ajax_nopriv_unlike_tribute_donation', $plugin_public, 'unlike_tribute_donation');

		$this->loader->add_action('wp_ajax_get_tribute_donations', $plugin_public, 'get_tribute_donations');
		$this->loader->add_action('wp_ajax_nopriv_get_tribute_donations', $plugin_public, 'get_tribute_donations');

		$this->loader->add_action('wp_ajax_add_donation_to_cart', $plugin_public, 'ajax_add_donation_to_cart');
		$this->loader->add_action('wp_ajax_nopriv_add_donation_to_cart', $plugin_public, 'ajax_add_donation_to_cart');

		$this->loader->add_shortcode( 'show_crm_tributes_large_grid', $plugin_public, 'shortcode_crm_tributes_large_grid_display' );

		$this->loader->add_shortcode( 'show_crm_tributes_grid', $plugin_public, 'shortcode_crm_tributes_grid_display' );
		$this->loader->add_shortcode( 'show_crm_tribute', $plugin_public, 'shortcode_crm_tribute_display' );
		$this->loader->add_shortcode( 'show_crm_tributes_carousel', $plugin_public, 'shortcode_crm_tributes_carousel_display' );
		$this->loader->add_shortcode( 'show_crm_tributes_flower_delivery', $plugin_public, 'shortcode_crm_tributes_flower_delivery' );

		// Update for single tributes using the Yoast SEO plugin
		$this->loader->add_filter( 'wpseo_canonical', $plugin_public, 'update_yoast_single_tribute_canonical_url' );
		$this->loader->add_filter( 'wpseo_opengraph_url', $plugin_public, 'update_yoast_single_tribute_opengraph_url' );
		$this->loader->add_filter( 'wpseo_opengraph_title', $plugin_public, 'update_yoast_single_tribute_opengraph_title' );
		$this->loader->add_filter( 'wpseo_opengraph_image', $plugin_public, 'update_yoast_single_tribute_opengraph_image' );
		$this->loader->add_filter( 'wpseo_opengraph_desc', $plugin_public, 'update_yoast_single_tribute_opengraph_desc' );

		// Yoast SEO - Handling of sitemap
		$this->loader->add_filter( 'wpseo_sitemap_index', $plugin_public, 'yoast_add_to_sitemap_index' );
		// $this->loader->add_action('firehawk_tribute_sitemap_generate_hourly', $plugin_public, 'generate_tribute_sitemap');
		// $this->loader->add_action('wp', $plugin_public, 'initiate_generate_tribute_sitemap');
		$this->loader->add_filter( 'template_include', $plugin_public, 'tribute_sitemap_template_include' );
		$this->loader->add_filter( 'redirect_canonical', $plugin_public, 'tribute_sitemap_redirect_canonical_callback', 100, 2 );
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Fcrm_Tributes_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
